// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_READING_LIST_CORE_DUAL_READING_LIST_MODEL_H_
#define COMPONENTS_READING_LIST_CORE_DUAL_READING_LIST_MODEL_H_

#include <map>
#include <memory>

#include "base/memory/raw_ptr.h"
#include "base/memory/scoped_refptr.h"
#include "base/observer_list.h"
#include "base/sequence_checker.h"
#include "components/reading_list/core/reading_list_entry.h"
#include "components/reading_list/core/reading_list_model.h"
#include "components/reading_list/core/reading_list_model_impl.h"
#include "components/reading_list/core/reading_list_model_observer.h"
#include "url/gurl.h"

namespace reading_list {

// ReadingListModel implementation that is capable of providing a merged view of
// two underlying instances of ReadingListModel. For newly-created entries, the
// class determines internally and based on sign-in & sync state, which
// instance should be used. It is useful to support sync-the-transport use-cases
// where the user is signed in but has sync turned off: in this case the two
// data sources (local entries and entries server-side) should be treated
// independently under the hood, but an in-memory merged view can be presented
// to UI layers and generally feature integrations.
class DualReadingListModel : public ReadingListModel,
                             public ReadingListModelObserver {
 public:
  enum class StorageStateForTesting {
    kNotFound,
    kExistsInAccountModelOnly,
    kExistsInLocalOrSyncableModelOnly,
    kExistsInBothModels
  };

  DualReadingListModel(
      std::unique_ptr<ReadingListModelImpl> local_or_syncable_model,
      std::unique_ptr<ReadingListModelImpl> account_model);
  ~DualReadingListModel() override;

  // KeyedService implementation.
  void Shutdown() override;

  // ReadingListModel implementation.
  bool loaded() const override;
  base::WeakPtr<syncer::ModelTypeControllerDelegate> GetSyncControllerDelegate()
      override;
  base::WeakPtr<syncer::ModelTypeControllerDelegate>
  GetSyncControllerDelegateForTransportMode() override;
  bool IsPerformingBatchUpdates() const override;
  std::unique_ptr<ScopedReadingListBatchUpdate> BeginBatchUpdates() override;
  base::flat_set<GURL> GetKeys() const override;
  size_t size() const override;
  size_t unread_size() const override;
  size_t unseen_size() const override;
  void MarkAllSeen() override;
  bool DeleteAllEntries() override;
  scoped_refptr<const ReadingListEntry> GetEntryByURL(
      const GURL& gurl) const override;
  bool IsUrlSupported(const GURL& url) override;
  bool NeedsExplicitUploadToSyncServer(const GURL& url) const override;
  const ReadingListEntry& AddOrReplaceEntry(
      const GURL& url,
      const std::string& title,
      reading_list::EntrySource source,
      base::TimeDelta estimated_read_time) override;
  void RemoveEntryByURL(const GURL& url) override;
  void SetReadStatusIfExists(const GURL& url, bool read) override;
  void SetEntryTitleIfExists(const GURL& url,
                             const std::string& title) override;
  void SetEstimatedReadTimeIfExists(
      const GURL& url,
      base::TimeDelta estimated_read_time) override;
  void SetEntryDistilledStateIfExists(
      const GURL& url,
      ReadingListEntry::DistillationState state) override;
  void SetEntryDistilledInfoIfExists(const GURL& url,
                                     const base::FilePath& distilled_path,
                                     const GURL& distilled_url,
                                     int64_t distilation_size,
                                     base::Time distilation_time) override;
  void AddObserver(ReadingListModelObserver* observer) override;
  void RemoveObserver(ReadingListModelObserver* observer) override;

  // ReadingListModelObserver overrides.
  void ReadingListModelBeganBatchUpdates(
      const ReadingListModel* model) override;
  void ReadingListModelCompletedBatchUpdates(
      const ReadingListModel* model) override;
  void ReadingListModelLoaded(const ReadingListModel* model) override;
  void ReadingListWillRemoveEntry(const ReadingListModel* model,
                                  const GURL& url) override;
  void ReadingListDidRemoveEntry(const ReadingListModel* model,
                                 const GURL& url) override;
  void ReadingListWillAddEntry(const ReadingListModel* model,
                               const ReadingListEntry& entry) override;
  void ReadingListDidAddEntry(const ReadingListModel* model,
                              const GURL& url,
                              reading_list::EntrySource source) override;
  void ReadingListDidApplyChanges(ReadingListModel* model) override;

  class ScopedReadingListBatchUpdateImpl : public ScopedReadingListBatchUpdate {
   public:
    ScopedReadingListBatchUpdateImpl(
        std::unique_ptr<ScopedReadingListBatchUpdate>
            local_or_syncable_model_batch,
        std::unique_ptr<ScopedReadingListBatchUpdate> account_model_batch);
    ~ScopedReadingListBatchUpdateImpl() override;

   private:
    std::unique_ptr<ScopedReadingListBatchUpdate>
        local_or_syncable_model_batch_;
    std::unique_ptr<ScopedReadingListBatchUpdate> account_model_batch_;
  };

  StorageStateForTesting GetStorageStateForURLForTesting(const GURL& url);

 private:
  void NotifyObserversWithWillRemoveEntry(const GURL& url);
  void NotifyObserversWithDidRemoveEntry(const GURL& url);
  void NotifyObserversWithDidApplyChanges();

  const std::unique_ptr<ReadingListModelImpl> local_or_syncable_model_;
  const std::unique_ptr<ReadingListModelImpl> account_model_;

  // Indicates whether a ReadingListModelImpl::RemoveEntryByURL is currently
  // performing on `local_or_syncable_model_` and `account_model_`.
  bool ongoing_remove_entry_by_url_ = false;

  unsigned int current_batch_updates_count_ = 0;

  base::ObserverList<ReadingListModelObserver>::Unchecked observers_;

  SEQUENCE_CHECKER(sequence_checker_);
};

}  // namespace reading_list

#endif  // COMPONENTS_READING_LIST_CORE_DUAL_READING_LIST_MODEL_H_
