/*
 * \brief  Handle ACPI Smart Battery Subsystem devices
 * \author Alexander Boettcher
 *
 */

/*
 * Copyright (C) 2016-2017 Genode Labs GmbH
 *
 * This file is part of the Genode OS framework, which is distributed
 * under the terms of the GNU Affero General Public License version 3.
 */

class Battery : Acpica::Callback<Battery>
{
	private:

		Acpica::Reportstate * _report;
		ACPI_HANDLE _sb;

		/*
		 * ACPI spec - 10.2.2.1 _BIF (Battery Information)
		 * (alternatively 10.2.2.2 _BIX could be used)
		 */
		Acpica::Buffer<char [512]> _battery;
		Genode::String<16>         _battery_name;

		void _init_static_info()
		{
			ACPI_STATUS res = AcpiEvaluateObjectTyped(_sb, ACPI_STRING("_BIF"),
			                                          nullptr, &_battery,
			                                          ACPI_TYPE_PACKAGE);
			ACPI_OBJECT * obj = reinterpret_cast<ACPI_OBJECT *>(_battery.object);
			if (ACPI_FAILURE(res) || !obj || obj->Package.Count != 13) {
				Genode::error("failed   - '", __func__, "' _BIF res=", Genode::Hex(res));
				return;
			}

			Acpica::Buffer<char [8]> battery_name;
			res = AcpiGetName(_sb, ACPI_SINGLE_NAME, &battery_name);
			if (ACPI_FAILURE(res)) {
				_battery_name = Genode::String<16>("unknown");
			} else {
				_battery_name = Genode::String<16>(battery_name.object);
			}
		}

		void _info(Genode::Generator &g)
		{
			g.node("name", [&] { g.append_quoted(_battery_name.string()); });

			const char * node_name[] = {
				"powerunit", "design_capacity", "last_full_capacity",
				"technology", "voltage", "warning_capacity", "low_capacity",
				"granularity1", "granularity2", "serial", "model", "type",
				"oem"
			};

			ACPI_OBJECT * obj = reinterpret_cast<ACPI_OBJECT *>(_battery.object);

			if (sizeof(node_name) / sizeof(node_name[0]) != obj->Package.Count)
				return;

			for (unsigned i = 0; i < 9; i++) {
				ACPI_OBJECT * v = &obj->Package.Elements[i];

				g.node(node_name[i], [&] {
					if (v->Type != ACPI_TYPE_INTEGER) {
						g.append_quoted("unknown");
						return;
					}

					g.attribute("value", v->Integer.Value);

					if (i == 0)
						g.append_quoted(v->Integer.Value == 0 ? "mW/mWh" :
						                v->Integer.Value == 1 ? "mA/mAh" :
						                                        "unknown");
					if (i == 3)
						g.append_quoted(v->Integer.Value == 0 ? "primary" :
						                v->Integer.Value == 1 ? "secondary" :
						                                        "unknown");
				});
			}

			for (unsigned i = 9; i < obj->Package.Count; i++) {
				ACPI_OBJECT * v = &obj->Package.Elements[i];

				g.node(node_name[i], [&] {

					if (v->Type != ACPI_TYPE_STRING)
						return;

					g.append_quoted(v->String.Pointer);
				});
			}
		}

		void _status(Genode::Generator &g)
		{
			/* 10.2.2.6 _BST (Battery Status) */
			Acpica::Buffer<char [256]> dynamic;
			ACPI_STATUS res = AcpiEvaluateObjectTyped(_sb, ACPI_STRING("_BST"),
			                                          nullptr, &dynamic,
			                                          ACPI_TYPE_PACKAGE);
			ACPI_OBJECT * obj = reinterpret_cast<ACPI_OBJECT *>(dynamic.object);
			if (ACPI_FAILURE(res) || !obj ||
			    obj->Package.Count != 4) {
				Genode::error("failed   - '", __func__, "' _BST res=", Genode::Hex(res));
				return;
			}

			Acpica::Buffer<ACPI_OBJECT> sta;
			res = AcpiEvaluateObjectTyped(_sb, ACPI_STRING("_STA"), nullptr,
			                              &sta, ACPI_TYPE_INTEGER);
			if (ACPI_FAILURE(res)) {
				g.node("status", [&] { g.append_quoted("unknown"); });
			} else
				g.node("status", [&] {
					unsigned long long const val = sta.object.Integer.Value;
					g.attribute("value", val);
					/* see "6.3.7 _STA" for more human readable decoding */
					if (!(sta.object.Integer.Value & ACPI_STA_BATTERY_PRESENT))
						g.append_quoted("battery not present");
				});

			const char * node_name[] = {
				"state", "present_rate", "remaining_capacity",
				"present_voltage"
			};

			if (sizeof(node_name) / sizeof(node_name[0]) != obj->Package.Count)
				return;

			for (unsigned i = 0; i < obj->Package.Count; i++) {
				ACPI_OBJECT * v = &obj->Package.Elements[i];

				g.node(node_name[i], [&] {
					if (v->Type != ACPI_TYPE_INTEGER) {
						g.append_quoted("unknown");
						return;
					}

					g.attribute("value", v->Integer.Value);

					if (i != 0)
						return;

					if (v->Integer.Value & 0x1) g.append_quoted("discharging");
					if (v->Integer.Value & 0x2) g.append_quoted("charging");
					if (v->Integer.Value & 0x4) g.append_quoted("critical low");
				});
			}
		}

	public:

		Battery(void * report, ACPI_HANDLE sb)
		: _report(reinterpret_cast<Acpica::Reportstate *>(report)), _sb(sb)
		{
			_init_static_info();
			if (_report)
				_report->add_notify(this);
		}

		static ACPI_STATUS detect(ACPI_HANDLE sb, UINT32, void *m, void **)
		{
			Acpica::Main * main = reinterpret_cast<Acpica::Main *>(m);
			Battery * dev_obj = new (main->heap) Battery(main->report, sb);

			ACPI_STATUS res =
				AcpiInstallNotifyHandler(sb, ACPI_DEVICE_NOTIFY,
				                         Acpica::Callback<Battery>::handler,
				                         dev_obj);
			if (ACPI_FAILURE(res)) {
				Genode::error("failed   - '", __func__, "' "
				              "res=", Genode::Hex(res));
				delete dev_obj;
				return AE_OK;
			}

			Acpica::Buffer<char [8]> battery_name;
			res = AcpiGetName (sb, ACPI_SINGLE_NAME, &battery_name);
			if (ACPI_FAILURE(res)) {
				Genode::error("failed   - '", __func__, "' battery name "
				              "res=", Genode::Hex(res));
				return AE_OK;
			}

			Acpica::Buffer<ACPI_OBJECT> sta;
			res = AcpiEvaluateObjectTyped(sb, ACPI_STRING("_STA"), nullptr, &sta,
			                              ACPI_TYPE_INTEGER);
			if (ACPI_FAILURE(res)) {
				Genode::error("failed   - '", __func__, "' _STA "
				              "res=", Genode::Hex(res));
				return AE_OK;
			}

			/* ACPI spec - 10.2.2.1 _BIF (Battery Information) */
			Acpica::Buffer<char [512]> battery;
			res = AcpiEvaluateObjectTyped(sb, ACPI_STRING("_BIF"),
			                              nullptr, &battery,
			                              ACPI_TYPE_PACKAGE);
			ACPI_OBJECT * obj = reinterpret_cast<ACPI_OBJECT *>(battery.object);
			if (ACPI_FAILURE(res) || !obj) {
				Genode::error("failed   - '", __func__, "' _BIF "
				              "res=", Genode::Hex(res));
				return AE_OK;
			}

			unsigned long long const val = sta.object.Integer.Value;

			if (obj->Package.Count < 13 ||
			    obj->Package.Elements[0].Type != ACPI_TYPE_INTEGER ||
			    obj->Package.Elements[12].Type != ACPI_TYPE_STRING ||
			    obj->Package.Elements[11].Type != ACPI_TYPE_STRING ||
			    obj->Package.Elements[10].Type != ACPI_TYPE_STRING ||
			    obj->Package.Elements[9].Type != ACPI_TYPE_STRING)
			{
				Genode::log("detected - battery "
				            "'", Genode::Cstring(battery_name.object), "' - "
				            "unknown state (", Genode::Hex(val),
				            val & ACPI_STA_BATTERY_PRESENT ? "" : "(not present)",
				            ")");
				return AE_OK;
			}

			using Genode::Cstring;
			Genode::log("detected - battery "
			            "'", Cstring(battery_name.object), "' "
			            "type='",   Cstring(obj->Package.Elements[11].String.Pointer), "' "
			            "OEM='",    Cstring(obj->Package.Elements[12].String.Pointer), "' "
			            "state=",   Genode::Hex(val),
			                        val & ACPI_STA_BATTERY_PRESENT ? "" : "(not present)", " "
			            "model='",  Cstring(obj->Package.Elements[10].String.Pointer), "' "
			            "serial='", Cstring(obj->Package.Elements[9].String.Pointer), "'");

			return AE_OK;
		}

		/*
		 * Acpica::Callback<> interface
		 */

		void handle(ACPI_HANDLE sb, UINT32 value)
		{
			if (_report)
				_report->battery_event();
		}

		void generate(Genode::Generator &g)
		{
			_info(g);
			_status(g);
		}
};
