/*
 * \brief  Test for dynamic ROM
 * \author Johannes Schlatow
 * \date   2025-11-20
 *
 * This test exercises the signalling and update protocol of the dynamic ROM
 * session. It periodically signals the ROM client, which is then supposed
 * to request the current dataspace and thereby update the ROM content.
 * Whenever the time between the signal and the client-initiated ROM update
 * is too long, we know that the client missed a signal.
 */

/*
 * Copyright (C) 2025 Genode Labs GmbH
 *
 * This file is part of the Genode OS framework, which is distributed
 * under the terms of the GNU Affero General Public License version 3.
 */

/* Genode includes */
#include <base/component.h>
#include <base/attached_rom_dataspace.h>
#include <base/log.h>
#include <base/heap.h>
#include <base/session_object.h>
#include <root/component.h>
#include <timer_session/connection.h>
#include <os/dynamic_rom_session.h>
#include <trace/timestamp.h>
#include <report_session/connection.h>

namespace Test {
	class Main;
	class Consumer;
	using namespace Genode;
	template <typename>
	class Instrumented_signal_handler;
}

template <typename T>
class Test::Instrumented_signal_handler : public Signal_dispatcher_base
{
	
	private:

		Signal_context_capability _cap;
		Entrypoint &_ep;
		T  &_obj;
		void (T::*_member) ();

		/*
		 * Noncopyable
		 */
		Instrumented_signal_handler(Instrumented_signal_handler const &);
		Instrumented_signal_handler &operator = (Instrumented_signal_handler const &);

	public:

		/**
		 * Constructor
		 *
		 * \param ep          entrypoint managing this signal RPC
		 * \param obj,member  object and method to call when
		 *                    the signal occurs
		 */
		Instrumented_signal_handler(Entrypoint &ep, T &obj, void (T::*member)())
		:
			_cap(ep.manage(*this)), _ep(ep), _obj(obj), _member(member)
		{ }

		~Instrumented_signal_handler() { _ep.dissolve(*this); }

		/**
		 * Interface of Signal_dispatcher_base
		 */
		void dispatch(unsigned num) override
		{
			log("dispatching ", num, " signals");
			(_obj.*_member)();
		}

		operator Capability<Signal_context>() const { return _cap; }

};


struct Test::Consumer
{
	Env                      &_env;
	Timer::Connection         _timer  { _env };
	Rom_connection            _rom    { _env, "test" };
	Report::Connection        _report { _env, "test" };
	Signal_handler<Consumer>  _rom_handler { _env.ep(), *this, &Consumer::_handle_rom };
	unsigned                  _cnt { 0 };
	Trace::Timestamp          _tsc_per_ms { 0 };
	Trace::Timestamp          _start_ts { 0 };

	void _handle_rom()
	{
		Trace::Timestamp const ts = Trace::timestamp();
		_rom.dataspace();

		log(Trace::timestamp(), " - rom dataspace done");

		/* use RPC to trigger ROM update at producer  */
		_report.dataspace();

		log(Trace::timestamp(), " - report dataspace done");

		// while (((Trace::timestamp() - ts) / _tsc_per_ms) < 10) {
		// 	asm volatile ("nop");
		// }

		if (((ts - _start_ts) / _tsc_per_ms) > 30'000)
			_env.parent().exit(0);
	}

	Consumer(Env & env)
	: _env(env),
	  _start_ts(Trace::timestamp())
	{
		/* calibrate tsc */
		enum { SLEEP_MS = 1000 };
		_timer.msleep(SLEEP_MS);
		_tsc_per_ms = (Trace::timestamp() - _start_ts) / SLEEP_MS;

		_rom.sigh(_rom_handler);
		_handle_rom();
	}
};


struct Test::Main : Dynamic_rom_session::Producer
{
	Env &_env;
	Heap _heap { _env.ram(), _env.rm() };

	Timer::Connection _timer { _env };

	void _handle_timeout();

	void _handle_trigger()
	{
		if (_rom_root._session.constructed()) {
			// log(Trace::timestamp(), " - trigger update");
			_rom_root._session->trigger_update();
		}
	}

	Signal_handler<Main> _timeout_handler {
		_env.ep(), *this, &Main::_handle_timeout };

	Signal_handler<Main> _trigger_handler {
		_env.ep(), *this, &Main::_handle_trigger };

	/* Dynamic_rom_session::Producer API */
	void generate(Generator &) override;

	struct Report_session : Session_object<Report::Session>
	{
		enum { RAM_DS_SIZE = 16 };

		Env                     &_env;
		Main                    &_main;
		Attached_ram_dataspace   _ram_ds  { _env.ram(), _env.rm(), RAM_DS_SIZE };

		
		Report_session(Env             &env,
		               Resources const &resources,
		               Label     const &label,
		               Diag      const &diag,
		               Main            &main)
		:
			Session_object(env.ep(), resources, label, diag),
			_env  { env },
			_main { main }
		{
			copy_cstring(_ram_ds.local_addr<char>(), "<empty/>", RAM_DS_SIZE);
		}

		Dataspace_capability dataspace() override
		{
			/* spend some time */
			log(Trace::timestamp(), " Report_session::dataspace()");
			return static_cap_cast<Dataspace>(_ram_ds.cap());
		}

		void submit(size_t /* length */) override { }

		void response_sigh(Signal_context_capability /* sigh */) override { }

		size_t obtain_response() override { return RAM_DS_SIZE; }
	};

	struct Report_root : Root_component<Report_session, Genode::Single_client>
	{
		Env  &_env;
		Main &_main;

		Constructible<Report_session> _session { };

		Create_result _create_session(const char *args) override
		{
			_session.construct(
					_env, session_resources_from_args(args),
					session_label_from_args(args),
					session_diag_from_args(args), _main);

			return *_session;
		}

	public:

		Report_root(Env &env, Allocator &alloc, Main &main)
		:
			Root_component<Report_session, Genode::Single_client> { &env.ep().rpc_ep(), &alloc },
			_env  { env },
			_main { main }
		{ }
	} _report_root { _env, _heap, *this };

	struct Rom_root : Root_component<Dynamic_rom_session, Genode::Single_client>
	{
		Env  &_env;
		Main &_main;

		Constructible<Dynamic_rom_session> _session { };

		Create_result _create_session(const char *) override
		{
			_session.construct(_env.ep().rpc_ep(), _env.ram(), _env.rm(), _main, true);
			return *_session;
		}

		void _destroy_session(Dynamic_rom_session &) override {
			_session.destruct(); }
		
		Rom_root(Env &env, Allocator &md_alloc, Main &main)
		:
			Root_component<Dynamic_rom_session, Genode::Single_client>(&env.ep().rpc_ep(), &md_alloc),
			_env(env), _main(main)
		{ }

	} _rom_root { _env, _heap, *this };

	Main(Env &env)
	: Dynamic_rom_session::Producer("test"),
	  _env(env)
	{
		_timer.sigh(_timeout_handler);

		_env.parent().announce(_env.ep().manage(_rom_root));
		_env.parent().announce(_env.ep().manage(_report_root));
	}
};


void Test::Main::_handle_timeout()
{
	error("Consumer missed a signal");
	_env.parent().exit(1);
}


void Test::Main::generate(Generator & g)
{
	/* 1s timeout */
	_timer.trigger_once(1000*1000);

	g.attribute("value", 1);

	/* signal myself to trigger update */
	Signal_transmitter(_trigger_handler).submit();
}


/***************
 ** Component **
 ***************/

void Component::construct(Genode::Env &env)
{
	using namespace Genode;

	/*
	 * Read value 'consumer' attribute to decide whether to perform
	 * the consumer or producer role.
	 */
	static Attached_rom_dataspace config(env, "config");
	bool const is_consumer = config.node().attribute_value("consumer", false);

	if (is_consumer) {
		log("--- test-dynamic_rom consumer started ---");
		static Test::Consumer consumer(env);
	} else {
		log("--- test-dynamic_rom producer started ---");
		static Test::Main producer(env);
	}
}
