/*
 * \brief  Child representation
 * \author Norman Feske
 * \date   2018-01-23
 */

/*
 * Copyright (C) 2018 Genode Labs GmbH
 *
 * This file is part of the Genode OS framework, which is distributed
 * under the terms of the GNU Affero General Public License version 3.
 */

#ifndef _CHILD_H_
#define _CHILD_H_

/* Genode includes */
#include <util/list_model.h>
#include <base/service.h>
#include <base/node.h>
#include <os/reporter.h>
#include <depot/archive.h>

namespace Depot_deploy {

	using namespace Depot;

	static constexpr Generator::Max_depth MAX_NODE_DEPTH = { 20 };

	struct Child;
}


class Depot_deploy::Child : public List_model<Child>::Element
{
	public:

		using Name             = String<100>;
		using Binary_name      = String<80>;
		using Config_name      = String<80>;
		using Depot_rom_server = String<32>;
		using Launcher_name    = String<100>;

		struct Prio_levels
		{
			unsigned value;

			int min_priority() const
			{
				return (value > 0) ? -(int)(value - 1) : 0;
			}
		};

	private:

		Allocator &_alloc;

		Reconstructible<Buffered_node> _start_node;        /* from config */
		Constructible<Buffered_node>   _launcher_node { };
		Constructible<Buffered_node>   _pkg_node { };      /* from blueprint */

		/*
		 * State of the condition check for generating the start node of
		 * the child. I.e., if the child is complete and configured but
		 * a used server component is missing, we need to suppress the start
		 * node until the condition is satisfied.
		 */
		enum Condition { UNCHECKED, SATISFIED, UNSATISFIED };

		Condition _condition { UNCHECKED };

		Name const _name;

		bool _defined_by_launcher() const
		{
			/*
			 * If the <start> node lacks a 'pkg' attribute, we expect the
			 * policy to be defined by a launcher XML snippet.
			 */
			return _start_node.constructed() && !_start_node->has_attribute("pkg");
		}

		Archive::Path _config_pkg_path() const
		{
			if (_defined_by_launcher() && _launcher_node.constructed())
				return _launcher_node->attribute_value("pkg", Archive::Path());

			return _start_node->attribute_value("pkg", Archive::Path());
		}

		Launcher_name _launcher_name() const
		{
			if (!_defined_by_launcher())
				return Launcher_name();

			if (_start_node->has_attribute("launcher"))
				return _start_node->attribute_value("launcher", Launcher_name());

			return _start_node->attribute_value("name", Launcher_name());
		}

		void _with_launcher_node(auto const &fn) const
		{
			if (_launcher_node.constructed() && _defined_by_launcher())
				fn(*_launcher_node);
			else
				fn(Node());
		}

		/*
		 * The pkg-archive path of the current blueprint query, which may
		 * deviate from pkg path given in the config, once the config is
		 * updated.
		 */
		Archive::Path _blueprint_pkg_path = _config_pkg_path();

		/*
		 * Quota definitions obtained from the blueprint
		 */
		Number_of_bytes _pkg_ram_quota { 0 };
		unsigned long   _pkg_cap_quota { 0 };
		unsigned long   _pkg_cpu_quota { 0 };

		Binary_name _binary_name { };
		Config_name _config_name { };

		/*
		 * Set if the depot query for the child's blueprint failed.
		 */
		enum class State { UNKNOWN, PKG_INCOMPLETE, PKG_COMPLETE };

		State _state = State::UNKNOWN;

		bool _configured() const
		{
			return _pkg_node.constructed()
			   && (_config_pkg_path() == _blueprint_pkg_path);
		}

		inline void _gen_routes(Generator &, Node const &,
		                        Depot_rom_server const &,
		                        Depot_rom_server const &) const;

		static void _gen_provides_sub_node(Generator &g, Node const &service,
		                                   Node::Type const &node_type,
		                                   Service::Name  const &service_name)
		{
			if (service.type() == node_type)
				g.node("service", [&] {
					g.attribute("name", service_name); });
		}

		static void _gen_copy_of_sub_node(Generator &g, Node const &from_node,
		                                  Node::Type const &sub_node_type)
		{
			from_node.with_optional_sub_node(sub_node_type.string(),
				[&] (Node const &sub_node) {
					if (!g.append_node(sub_node, MAX_NODE_DEPTH))
						warning("sub node exceeds max depth: ", from_node); });
		}

	public:

		Child(Allocator &alloc, Node const &start_node)
		:
			_alloc(alloc),
			_start_node(_alloc, start_node),
			_name(_start_node->attribute_value("name", Name()))
		{ }

		Name name() const { return _name; }

		/*
		 * \return true if config had an effect on the child's state
		 */
		bool apply_config(Node const &start_node)
		{
			if (!start_node.differs_from(*_start_node))
				return false;

			Archive::Path const old_pkg_path = _config_pkg_path();

			/* import new start node */
			_start_node.construct(_alloc, start_node);

			Archive::Path const new_pkg_path = _config_pkg_path();

			/* invalidate blueprint if 'pkg' path changed */
			if (old_pkg_path != new_pkg_path) {
				_blueprint_pkg_path = new_pkg_path;
				_pkg_node.destruct();

				/* reset error state, attempt to obtain the blueprint again */
				_state = State::UNKNOWN;
			}
			return true;
		}

		/*
		 * \return true if bluerprint had an effect on the child
		 */
		bool apply_blueprint(Node const &pkg)
		{
			if (_state == State::PKG_COMPLETE)
				return false;

			if (pkg.attribute_value("path", Archive::Path()) != _blueprint_pkg_path)
				return false;

			/* check for the completeness of all ROM ingredients */
			bool any_rom_missing = false;
			pkg.for_each_sub_node("missing_rom", [&] (Node const &missing_rom) {
				Name const name = missing_rom.attribute_value("label", Name());

				/* ld.lib.so is special because it is provided by the base system */
				if (name == "ld.lib.so")
					return;

				warning("missing ROM module '", name, "' needed by ", _blueprint_pkg_path);
				any_rom_missing = true;
			});

			if (any_rom_missing) {
				State const orig_state = _state;
				_state = State::PKG_INCOMPLETE;
				return (orig_state != _state);
			}

			/* package was missing but is installed now */
			_state = State::PKG_COMPLETE;

			pkg.with_sub_node("runtime",
				[&] (Node const &runtime) {

					_pkg_ram_quota = runtime.attribute_value("ram", Number_of_bytes());
					_pkg_cap_quota = runtime.attribute_value("caps", 0UL);
					_pkg_cpu_quota = runtime.attribute_value("cpu", 0UL);

					_binary_name = runtime.attribute_value("binary", Binary_name());
					_config_name = runtime.attribute_value("config", Config_name());

					/* keep copy of the blueprint info */
					_pkg_node.construct(_alloc, pkg);
				},
				[&] { });

			return true;
		}

		bool apply_launcher(Launcher_name const &name, Node const &launcher)
		{
			if (!_defined_by_launcher())
				return false;

			if (_launcher_name() != name)
				return false;

			if (_launcher_node.constructed() && !launcher.differs_from(*_launcher_node))
				return false;

			_launcher_node.construct(_alloc, launcher);

			_blueprint_pkg_path = _config_pkg_path();

			return true;
		}

		/*
		 * \return true if condition changed
		 */
		bool apply_condition(auto const &cond_fn)
		{
			Condition const orig_condition = _condition;

			_with_launcher_node([&] (Node const &launcher_node) {
				if (_start_node.constructed())
					_condition = cond_fn(*_start_node, launcher_node)
					           ? SATISFIED : UNSATISFIED;
			});

			return _condition != orig_condition;
		}

		void apply_if_unsatisfied(auto const &fn) const
		{
			_with_launcher_node([&] (Node const &launcher_node) {
				if (_condition == UNSATISFIED && _start_node.constructed())
					fn(*_start_node, launcher_node, _name);
			});
		}

		/*
		 * \return true if the call had an effect on the child
		 */
		bool mark_as_incomplete(Node const &missing)
		{
			/* print error message only once */
			if(_state == State::PKG_INCOMPLETE)
				return false;

			Archive::Path const path = missing.attribute_value("path", Archive::Path());
			if (path != _blueprint_pkg_path)
				return false;

			log(path, " incomplete or missing");

			State const orig_state = _state;
			_state = State::PKG_INCOMPLETE;

			return (orig_state != _state);
		}

		/**
		 * Reconsider deployment of child after installing missing archives
		 */
		void reset_incomplete()
		{
			if (_state == State::PKG_INCOMPLETE) {
				_state = State::UNKNOWN;
				_pkg_node.destruct();
			}
		}

		bool blueprint_needed() const
		{
			if (_configured())
				return false;

			if (_defined_by_launcher() && !_launcher_node.constructed())
				return false;

			return true;
		}

		void gen_query(Generator &g) const
		{
			if (blueprint_needed())
				g.node("blueprint", [&] {
					g.attribute("pkg", _blueprint_pkg_path); });
		}

		/**
		 * Generate start node of init configuration
		 *
		 * \param common              session routes to be added in addition to
		 *                            the ones found in the pkg blueprint
		 * \param cached_depot_rom    name of the server that provides the depot
		 *                            content as ROM modules. If the string is
		 *                            invalid, ROM requests are routed to the
		 *                            parent.
		 * \param uncached_depot_rom  name of the depot-ROM server used to obtain
		 *                            the content of the depot user "local", which
		 *                            is assumed to be mutable
		 */
		inline void gen_start_node(Generator              &,
		                           Node             const &common,
		                           Prio_levels             prio_levels,
		                           Affinity::Space         affinity_space,
		                           Depot_rom_server const &cached_depot_rom,
		                           Depot_rom_server const &uncached_depot_rom) const;

		inline void gen_monitor_policy_node(Generator &) const;

		void with_missing_pkg_path(auto const &fn) const
		{
			if (_state == State::PKG_INCOMPLETE)
				fn(_config_pkg_path());
		}

		/**
		 * Generate installation entry needed for the completion of the child
		 */
		void gen_installation_entry(Generator &g) const
		{
			with_missing_pkg_path([&] (Archive::Path const &path) {
				g.node("archive", [&] {
					g.attribute("path", path);
					g.attribute("source", "no"); }); });
		}

		bool incomplete() const { return _state == State::PKG_INCOMPLETE; }

		/**
		 * List_model::Element
		 */
		bool matches(Node const &node) const
		{
			return node.attribute_value("name", Child::Name()) == _name;
		}

		/**
		 * List_model::Element
		 */
		static bool type_matches(Node const &node)
		{
			return node.has_type("start");
		}
};


void Depot_deploy::Child::gen_start_node(Generator              &g,
                                         Node             const &common,
                                         Prio_levels      const  prio_levels,
                                         Affinity::Space  const  affinity_space,
                                         Depot_rom_server const &cached_depot_rom,
                                         Depot_rom_server const &uncached_depot_rom) const
{
	if (!_configured() || _condition == UNSATISFIED)
		return;

	if (_defined_by_launcher() && !_launcher_node.constructed())
		return;

	if (!_pkg_node->has_sub_node("runtime")) {
		warning("blueprint for '", _name, "' lacks runtime information");
		return;
	}

	Node const &start_node = *_start_node;

	g.node("start", [&] {

		g.attribute("name", _name);

		{
			unsigned long caps = _pkg_cap_quota;
			_with_launcher_node([&] (Node const &launcher_node) {
				caps = launcher_node.attribute_value("caps", caps); });
			caps = start_node.attribute_value("caps", caps);
			g.attribute("caps", caps);
		}

		{
			using Version = String<64>;
			Version const version = _start_node->attribute_value("version", Version());
			if (version.valid())
				g.attribute("version", version);
		}

		{
			long priority = prio_levels.min_priority();
			_with_launcher_node([&] (Node const &launcher_node) {
				priority = launcher_node.attribute_value("priority", priority); });
			priority = start_node.attribute_value("priority", priority);
			if (priority)
				g.attribute("priority", priority);
		}

		auto permit_managing_system = [&]
		{
			bool result = false;

			if (start_node.attribute_value("managing_system", false))
				result = true;

			_with_launcher_node([&] (Node const &launcher_node) {
				if (launcher_node.attribute_value("managing_system", false))
					result = true; });

			return result;
		};
		if (permit_managing_system())
			g.attribute("managing_system", "yes");

		bool shim_reroute = false;

		/* lookup if PD/CPU service is configured and use shim in such cases */
		start_node.with_optional_sub_node("route", [&] (Node const &route) {
			route.for_each_sub_node("service", [&] (Node const &service) {
				if (service.attribute_value("name", Name()) == "PD" ||
				    service.attribute_value("name", Name()) == "CPU")
					shim_reroute = true; }); });

		Binary_name const binary = shim_reroute ? Binary_name("shim")
		                                        : _binary_name;

		g.node("binary", [&] { g.attribute("name", binary); });

		Number_of_bytes ram = _pkg_ram_quota;
		_with_launcher_node([&] (Node const &launcher_node) {
			ram = launcher_node.attribute_value("ram", ram); });
		ram = start_node.attribute_value("ram", ram);

		g.node("resource", [&] {
			g.attribute("name", "RAM");
			g.attribute("quantum", String<32>(ram));
		});

		unsigned long cpu_quota = _pkg_cpu_quota;
		_with_launcher_node([&] (Node const &launcher_node) {
			cpu_quota = launcher_node.attribute_value("cpu", cpu_quota); });
		cpu_quota = start_node.attribute_value("cpu", cpu_quota);

		g.node("resource", [&] {
			g.attribute("name", "CPU");
			g.attribute("quantum", cpu_quota);
		});

		/* affinity-location handling */
		bool affinity_from_launcher = false;
		_with_launcher_node([&] (Node const &launcher_node) {
			affinity_from_launcher = launcher_node.has_sub_node("affinity"); });

		bool const affinity_from_start = start_node.has_sub_node("affinity");

		if (affinity_from_start || affinity_from_launcher) {

			Affinity::Location location { };

			if (affinity_from_launcher)
				_with_launcher_node([&] (Node const &launcher_node) {
					launcher_node.with_optional_sub_node("affinity", [&] (Node const &node) {
						location = Affinity::Location::from_node(affinity_space, node); }); });

			if (affinity_from_start)
				start_node.with_optional_sub_node("affinity", [&] (Node const &node) {
					location = Affinity::Location::from_node(affinity_space, node); });

			g.node("affinity", [&] {
				g.attribute("xpos",   location.xpos());
				g.attribute("ypos",   location.ypos());
				g.attribute("width",  location.width());
				g.attribute("height", location.height());
			});
		}

		/* runtime handling */
		_pkg_node->with_optional_sub_node("runtime", [&] (Node const &runtime) {

			/*
			 * Insert inline '<heartbeat>' node if provided by the start node
			 * or launcher.
			 */
			if (start_node.has_sub_node("heartbeat"))
				_gen_copy_of_sub_node(g, start_node, "heartbeat");
			else
				_with_launcher_node([&] (Node const &launcher_node) {
					if (launcher_node.has_sub_node("heartbeat"))
						_gen_copy_of_sub_node(g, launcher_node, "heartbeat"); });

			/*
			 * Insert inline '<config>' node if provided by the start node,
			 * the launcher definition (if a launcher is used), or the
			 * blueprint. The former is preferred over the latter.
			 */
			bool config_defined = false;
			if (start_node.has_sub_node("config")) {
				_gen_copy_of_sub_node(g, start_node, "config");
				config_defined = true; }

			if (!config_defined)
				_with_launcher_node([&] (Node const &launcher_node) {
					if (launcher_node.has_sub_node("config")) {
						_gen_copy_of_sub_node(g, launcher_node, "config");
						config_defined = true; } });

			if (!config_defined)
				if (runtime.has_sub_node("config")) {
					_gen_copy_of_sub_node(g, runtime, "config");
					config_defined = true; }

			/*
			 * Declare services provided by the subsystem.
			 */
			runtime.with_optional_sub_node("provides", [&] (Node const &provides) {
				g.node("provides", [&] {
					provides.for_each_sub_node([&] (Node const &service) {
						_gen_provides_sub_node(g, service, "audio_in",    "Audio_in");
						_gen_provides_sub_node(g, service, "audio_out",   "Audio_out");
						_gen_provides_sub_node(g, service, "block",       "Block");
						_gen_provides_sub_node(g, service, "file_system", "File_system");
						_gen_provides_sub_node(g, service, "framebuffer", "Framebuffer");
						_gen_provides_sub_node(g, service, "input",       "Input");
						_gen_provides_sub_node(g, service, "event",       "Event");
						_gen_provides_sub_node(g, service, "log",         "LOG");
						_gen_provides_sub_node(g, service, "nic",         "Nic");
						_gen_provides_sub_node(g, service, "uplink",      "Uplink");
						_gen_provides_sub_node(g, service, "gui",         "Gui");
						_gen_provides_sub_node(g, service, "gpu",         "Gpu");
						_gen_provides_sub_node(g, service, "usb",         "Usb");
						_gen_provides_sub_node(g, service, "report",      "Report");
						_gen_provides_sub_node(g, service, "rom",         "ROM");
						_gen_provides_sub_node(g, service, "terminal",    "Terminal");
						_gen_provides_sub_node(g, service, "timer",       "Timer");
						_gen_provides_sub_node(g, service, "pd",          "PD");
						_gen_provides_sub_node(g, service, "cpu",         "CPU");
						_gen_provides_sub_node(g, service, "rtc",         "Rtc");
						_gen_provides_sub_node(g, service, "capture",     "Capture");
						_gen_provides_sub_node(g, service, "play",        "Play");
						_gen_provides_sub_node(g, service, "record",      "Record");
					});
				});
			});

			g.tabular_node("route", [&] {

				if (start_node.has_sub_node("monitor")) {
					g.node("service", [&] {
						g.attribute("name", "PD");
						g.node("local");
					});
					g.node("service", [&] {
						g.attribute("name", "CPU");
						g.node("local");
					});
					g.node("service", [&] {
						g.attribute("name", "VM");
						g.node("local");
					});
				}

				_gen_routes(g, common, cached_depot_rom, uncached_depot_rom);
			});
		});
	});
}


void Depot_deploy::Child::gen_monitor_policy_node(Generator &g) const
{
	if (!_configured() || _condition == UNSATISFIED)
		return;

	if (_defined_by_launcher() && !_launcher_node.constructed())
		return;

	if (!_pkg_node->has_sub_node("runtime")) {
		return;
	}

	_start_node->with_optional_sub_node("monitor", [&] (Node const &monitor) {
		g.node("policy", [&] {
			g.attribute("label", _name);
			g.attribute("wait", monitor.attribute_value("wait", false));
			g.attribute("wx",   monitor.attribute_value("wx",   false));
		});
	});
}


void Depot_deploy::Child::_gen_routes(Generator &g, Node const &common,
                                      Depot_rom_server const &cached_depot_rom,
                                      Depot_rom_server const &uncached_depot_rom) const
{
	bool route_binary_to_shim = false;

	if (!_pkg_node.constructed())
		return;

	using Path = String<160>;

	auto copy_route = [&] (Node const &service)
	{
		g.node(service.type().string(), [&] {
			g.node_attributes(service);
			service.for_each_sub_node([&] (Node const &target) {
				g.node(target.type().string(), [&] {
					g.node_attributes(target); }); }); });
	};

	/*
	 * Add routes given in the start node.
	 */
	_start_node->with_optional_sub_node("route", [&] (Node const &route) {

		route.for_each_sub_node("service", [&] (Node const &service) {
			Name const service_name = service.attribute_value("name", Name());

			/* supplement env-session routes for the shim */
			if (service_name == "PD" || service_name == "CPU") {
				route_binary_to_shim = true;

				g.node("service", [&] {
					g.attribute("name", service_name);
					g.attribute("unscoped_label", _name);
					g.node("parent", [&] { });
				});
			}

			copy_route(service);
		});
	});

	/*
	 * If the subsystem is hosted under a shim, make the shim binary available
	 */
	if (route_binary_to_shim)
		g.node("service", [&] {
			g.attribute("name", "ROM");
			g.attribute("unscoped_label", "shim");
			g.node("parent", [&] {
				g.attribute("label", "shim"); }); });

	/*
	 * Add routes given in the launcher definition.
	 */
	_with_launcher_node([&] (Node const &launcher) {
		launcher.with_optional_sub_node("route", [&] (Node const &route) {
			route.for_each_sub_node([&] (Node const &service) {
				copy_route(service); }); }); });

	/**
	 * Return name of depot-ROM server used for obtaining the 'path'
	 *
	 * If the depot path refers to the depot-user "local", route the
	 * session request to the non-cached ROM service.
	 */
	auto rom_server = [&] (Path const &path) {

		return (String<7>(path) == "local/") ? uncached_depot_rom
		                                     : cached_depot_rom;
	};

	/*
	 * Redirect config ROM request to label as given in the 'config' attribute,
	 * if present. We need to search the blueprint's <rom> nodes for the
	 * matching ROM module to rewrite the label with the configuration's path
	 * within the depot.
	 */
	if (_config_name.valid()) {
		_pkg_node->for_each_sub_node("rom", [&] (Node const &rom) {

			if (!rom.has_attribute("path"))
				return;

			if (rom.attribute_value("label", Name()) != _config_name)
				return;

			/* we found the <rom> node for the config ROM */
			g.node("service", [&] {
				g.attribute("name",  "ROM");
				g.attribute("label", "config");
				using Path = String<160>;
				Path const path = rom.attribute_value("path", Path());

				if (cached_depot_rom.valid())
					g.node("child", [&] {
						g.attribute("name", rom_server(path));
						g.attribute("label", path); });
				else
					g.node("parent", [&] {
						g.attribute("label", path); });
			});
		});
	}

	/*
	 * Add common routes as defined in our config.
	 */
	common.for_each_sub_node([&] (Node const &service) {
		copy_route(service); });

	/*
	 * Add ROM routing rule with the label rewritten to the path within the
	 * depot.
	 */
	_pkg_node->for_each_sub_node("rom", [&] (Node const &rom) {

		if (!rom.has_attribute("path"))
			return;

		using Label = Name;
		Path  const path  = rom.attribute_value("path",  Path());
		Label const label = rom.attribute_value("label", Label());
		Label const as    = rom.attribute_value("as",    label);

		g.node("service", [&] {
			g.attribute("name", "ROM");

			if (route_binary_to_shim && label == _binary_name)
				g.attribute("label", "binary");
			else
				g.attribute("label_last", as);

			if (cached_depot_rom.valid()) {
				g.node("child", [&] {
					g.attribute("name",  rom_server(path));
					g.attribute("label", path);
				});
			} else {
				g.node("parent", [&] {
					g.attribute("label", path); });
			}
		});
	});
}

#endif /* _CHILD_H_ */
