/********************************************************************/
/*                                                                  */
/*  soc_rtl.c     Primitive actions for the socket type.            */
/*  Copyright (C) 1989 - 2015, 2018 - 2021  Thomas Mertes           */
/*                                                                  */
/*  This file is part of the Seed7 Runtime Library.                 */
/*                                                                  */
/*  The Seed7 Runtime Library is free software; you can             */
/*  redistribute it and/or modify it under the terms of the GNU     */
/*  Lesser General Public License as published by the Free Software */
/*  Foundation; either version 2.1 of the License, or (at your      */
/*  option) any later version.                                      */
/*                                                                  */
/*  The Seed7 Runtime Library is distributed in the hope that it    */
/*  will be useful, but WITHOUT ANY WARRANTY; without even the      */
/*  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR */
/*  PURPOSE.  See the GNU Lesser General Public License for more    */
/*  details.                                                        */
/*                                                                  */
/*  You should have received a copy of the GNU Lesser General       */
/*  Public License along with this program; if not, write to the    */
/*  Free Software Foundation, Inc., 51 Franklin Street,             */
/*  Fifth Floor, Boston, MA  02110-1301, USA.                       */
/*                                                                  */
/*  Module: Seed7 Runtime Library                                   */
/*  File: seed7/src/soc_rtl.c                                       */
/*  Changes: 2007, 2011, 2013 - 2015, 2018 - 2020  Thomas Mertes    */
/*  Content: Primitive actions for the socket type.                 */
/*                                                                  */
/********************************************************************/

#define LOG_FUNCTIONS 0
#define VERBOSE_EXCEPTIONS 0

#include "version.h"

#if SOCKET_LIB == UNIX_SOCKETS || SOCKET_LIB == WINSOCK_SOCKETS
#include "stdlib.h"
#include "stdio.h"
#include "string.h"
#include "limits.h"

#if SOCKET_LIB == UNIX_SOCKETS

#include "sys/types.h"
#include "sys/socket.h"
#include "netdb.h"
#include "netinet/in.h"
#if HAS_SELECT
#include "sys/select.h"
#endif
#if HAS_POLL
#include "poll.h"
#endif

#elif SOCKET_LIB == WINSOCK_SOCKETS

#define FD_SETSIZE 16384
#include "winsock2.h"
#if HAS_GETADDRINFO
#include "ws2tcpip.h"
#endif

#endif

#include "errno.h"

#if UNISTD_H_PRESENT
#include "unistd.h"
#endif

#include "common.h"
#include "data_rtl.h"
#include "heaputl.h"
#include "striutl.h"
#include "os_decls.h"
#include "int_rtl.h"
#include "rtl_err.h"

#undef EXTERN
#define EXTERN
#include "soc_rtl.h"


#if SOCKET_LIB == UNIX_SOCKETS

typedef socklen_t sockLenType;
#define check_initialization(err_result)
#define cast_send_recv_data(data_ptr) ((void *) (data_ptr))
#define cast_buffer_len(len)          len
#define ADDRLEN_NEGATIVE(len) 0
#define INVALID_SOCKET (-1)

#elif SOCKET_LIB == WINSOCK_SOCKETS

typedef int sockLenType;
static boolType initialized = FALSE;
#define check_initialization(err_result) \
    if (unlikely(!initialized)) {if (init_winsock()) {return err_result;}}
#define cast_send_recv_data(data_ptr) ((char *) (data_ptr))
#define cast_buffer_len(len)          ((int) (len))
#define ADDRLEN_NEGATIVE(len) (len) < 0
#ifndef SHUT_RDWR
#define SHUT_RDWR SD_BOTH
#endif

#endif

#define BUFFER_SIZE                4096
#define GETS_DEFAULT_SIZE       1048576
#define READ_STRI_INIT_SIZE         256
#define READ_STRI_SIZE_DELTA       2048
#define MAX_ADDRESS_SIZE           1024
#define MAX_HOSTNAME_LENGTH        1024
#define ERROR_MESSAGE_BUFFER_SIZE  1024
#define SHOW_ADDRINFO 0

#define MAX_SOCK_ADDRESS_LEN \
    STRLEN("[0123:4567:89ab:cdef:0123:4567:89ab:cdef]:") + UINT16TYPE_DECIMAL_SIZE



#if ANY_LOG_ACTIVE
static const_cstriType socAddressCStri (const const_bstriType address)

  {
    const struct sockaddr *addr;
    static char buffer[MAX_SOCK_ADDRESS_LEN + NULL_TERMINATION_LEN];
    cstriType result;

  /* socAddressCStri */
    if (unlikely(address == NULL)) {
      result = " *NULL_ADDRESS* ";
    } else if (unlikely(address->size == 0)) {
      result = " *EMPTY_ADDRESS* ";
    } else if (unlikely(address->size < sizeof(struct sockaddr))) {
      result = " ** Size of address too small ** ";
    } else {
      addr = (const struct sockaddr *) address->mem;
      switch (addr->sa_family) {
        case AF_INET:
          if (unlikely(address->size != sizeof(struct sockaddr_in))) {
            result = " ** Size of address wrong for AF_INET ** ";
          } else {
            const struct sockaddr_in *inet_address = (const struct sockaddr_in *) address->mem;
            uint32Type ip4_address = ntohl(inet_address->sin_addr.s_addr);
            uint16Type port;
            port = ntohs(inet_address->sin_port);       /* short, network byte order */

            sprintf(buffer, "%d.%d.%d.%d:" FMT_U16,
                (ip4_address >> 24) & 255,
                (ip4_address >> 16) & 255,
                (ip4_address >>  8) & 255,
                 ip4_address        & 255, port);
            result = buffer;
          } /* if */
          break;
#if HAS_GETADDRINFO || defined INET6_SERVER_ADDRESS
        case AF_INET6:
          if (unlikely(address->size != sizeof(struct sockaddr_in6))) {
            result = " ** Size of address wrong for AF_INET6 ** ";
          } else {
            const struct sockaddr_in6 *inet6_address =
                (const struct sockaddr_in6 *) address->mem;
            unsigned int digitGroup[8];
            int pos;
            uint16Type port;

            for (pos = 0; pos <= 7; pos++) {
              digitGroup[pos] =
                  (unsigned int) (inet6_address->sin6_addr.s6_addr[pos << 1]) << 8 |
                  (unsigned int) (inet6_address->sin6_addr.s6_addr[(pos << 1) + 1]);
            } /* for */
            port = ntohs(inet6_address->sin6_port);     /* short, network byte order */

            sprintf(buffer, "[%x:%x:%x:%x:%x:%x:%x:%x]:" FMT_U16,
                digitGroup[0], digitGroup[1], digitGroup[2], digitGroup[3],
                digitGroup[4], digitGroup[5], digitGroup[6], digitGroup[7], port);
            result = buffer;
          } /* if */
          break;
#endif
        default:
          result = " ** Address neither AF_INET nor AF_INET6 ** ";
          break;
      } /* switch */
    } /* if */
    return result;
  } /* socAddressCStri */
#endif



#if SOCKET_LIB == WINSOCK_SOCKETS
static int init_winsock (void)

  {
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;
    int result;

  /* init_winsock */
    wVersionRequested = MAKEWORD(2, 2);
    err = WSAStartup(wVersionRequested, &wsaData);
    if (err != 0) {
      logError(printf("init_winsock: WSAStartup() failed:\n"
                      "%s=%d\nerror: %s\n",
                      ERROR_INFORMATION););
      raise_error(FILE_ERROR);
      result = -1;
    } else {
      if (LOBYTE(wsaData.wVersion) != 2 ||
          HIBYTE(wsaData.wVersion) != 2) {
        WSACleanup();
        logError(printf("init_winsock: WSAStartup() "
                        "did not return the requested version:\n"););
        raise_error(FILE_ERROR);
        result = -1;
      } else {
        initialized = TRUE;
        result = 0;
      } /* if */
    } /* if */
    return result;
  } /* init_winsock */



const_cstriType wsaErrorMessage (void)

  {
    static char buffer[ERROR_MESSAGE_BUFFER_SIZE];
    const_cstriType errorMessage = buffer;

  /* wsaErrorMessage */
    if (FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
                       NULL, WSAGetLastError(),
                       MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                       buffer, ERROR_MESSAGE_BUFFER_SIZE, NULL) == 0) {
      errorMessage = "Unable to get error message.";
    } /* if */
    return errorMessage;
  } /* wsaErrorMessage */
#endif



#if HAS_GETADDRINFO
static struct addrinfo *select_addrinfo (struct addrinfo *addrinfo_list,
    int addr_family1, int addr_family2)

  {
    struct addrinfo *current_addrinfo;
    struct addrinfo *inet_addrinfo = NULL;
    struct addrinfo *inet6_addrinfo = NULL;
    struct addrinfo *alternate_inet_addrinfo = NULL;
    struct addrinfo *result_addrinfo = NULL;

  /* select_addrinfo */
    current_addrinfo = addrinfo_list;
    while (current_addrinfo != NULL) {
      if (current_addrinfo->ai_addr->sa_family == AF_INET) {
        if (inet_addrinfo == NULL) {
          struct sockaddr_in *inet_address = (struct sockaddr_in *) current_addrinfo->ai_addr;
          uint32Type ip4_address = ntohl(inet_address->sin_addr.s_addr);
          /* printf("ip4=%d.%d.%d.%d\n",
              ip4_address >> 24, (ip4_address >> 16) & 255,
              (ip4_address >> 8) & 255, ip4_address & 255); */
          if ((ip4_address & (127 << 24)) == (127 << 24)) {
            alternate_inet_addrinfo = current_addrinfo;
          } else {
            inet_addrinfo = current_addrinfo;
          } /* if */
        } /* if */
      } else if (current_addrinfo->ai_addr->sa_family == AF_INET6) {
        if (inet6_addrinfo == NULL) {
          inet6_addrinfo = current_addrinfo;
        } /* if */
      } /* if */
      current_addrinfo = current_addrinfo->ai_next;
    } /* while */
    if (inet_addrinfo == NULL) {
      inet_addrinfo = alternate_inet_addrinfo;
    } /* if */
    if (addr_family1 == AF_INET && inet_addrinfo != NULL) {
      result_addrinfo = inet_addrinfo;
    } else if (addr_family1 == AF_INET6 && inet6_addrinfo != NULL) {
      result_addrinfo = inet6_addrinfo;
    } /* if */
    if (result_addrinfo == NULL) {
      if (addr_family2 == AF_INET && inet_addrinfo != NULL) {
        result_addrinfo = inet_addrinfo;
      } else if (addr_family2 == AF_INET6 && inet6_addrinfo != NULL) {
        result_addrinfo = inet6_addrinfo;
      } /* if */
      if (result_addrinfo == NULL) {
        /* No addr_family1 and addr_family2 present: Take first address */
        result_addrinfo = addrinfo_list;
      } /* if */
    } /* if */
    return result_addrinfo;
  } /* select_addrinfo */
#endif



#if SHOW_ADDRINFO
static void showAddrinfo (struct addrinfo *addrinfo_list)

  {
    struct addrinfo *addr;

  /* showAddrinfo */
    addr = addrinfo_list;
    do {
      printf("ai_flags=%d\n",    addr->ai_flags);
      printf("ai_family=%d (%s)\n", addr->ai_family,
          addr->ai_family == AF_INET ? "AF_INET" :
          (addr->ai_family == AF_INET6 ? "AF_INET6" :
          (addr->ai_family == AF_UNSPEC ? "AF_UNSPEC" : "UNKNOWN")));
      printf("ai_socktype=%d\n", addr->ai_socktype);
      printf("ai_protocol=%d\n", addr->ai_protocol);
      printf("ai_addrlen=%d\n", addr->ai_addrlen);
      printf("sa_family=%d (%s)\n", addr->ai_addr->sa_family,
          addr->ai_addr->sa_family == AF_INET ? "AF_INET" :
          (addr->ai_addr->sa_family == AF_INET6 ? "AF_INET6" :
          (addr->ai_addr->sa_family == AF_UNSPEC ? "AF_UNSPEC" : "UNKNOWN")));
      if (addr->ai_addr->sa_family == AF_INET) {
        struct sockaddr_in *inet_address = (struct sockaddr_in *) addr->ai_addr;
        uint32Type ip4_address = ntohl(inet_address->sin_addr.s_addr);
        printf("sin_port=%d\n", ntohs(inet_address->sin_port));
        printf("sin_addr.s_addr=%d.%d.%d.%d\n",
            (ip4_address >> 24) & 255,
            (ip4_address >> 16) & 255,
            (ip4_address >>  8) & 255,
             ip4_address        & 255);
      } else if (addr->ai_addr->sa_family == AF_INET6) {
        struct sockaddr_in6 *inet6_address = (struct sockaddr_in6 *) addr->ai_addr;
        unsigned int digitGroup[8];
        int pos;
        printf("sin_port=%d\n", ntohs(inet6_address->sin6_port));
        for (pos = 0; pos <= 7; pos++) {
          digitGroup[pos] =
              (unsigned int) inet6_address->sin6_addr.s6_addr[pos << 1] << 8 |
              (unsigned int) inet6_address->sin6_addr.s6_addr[(pos << 1) + 1];
        } /* for */
        printf("sin_addr.s_addr=%x:%x:%x:%x:%x:%x:%x:%x\n",
            digitGroup[0], digitGroup[1], digitGroup[2], digitGroup[3],
            digitGroup[4], digitGroup[5], digitGroup[6], digitGroup[7]);
      }
      printf("ai_canonname=%s\n", addr->ai_canonname);
      printf("ai_next=%lx\n", (memSizeType) addr->ai_next);
      logMessage({
                   char name[MAX_HOSTNAME_LENGTH];
                   char serv[MAX_ADDRESS_SIZE];
                   getnameinfo(addr->ai_addr, 100, name, MAX_HOSTNAME_LENGTH,
                               serv, MAX_ADDRESS_SIZE, 0);
                   printf("name=%s\n", name);
                   printf("serv=%s\n", serv);
                 });
      addr = addr->ai_next;
    } while (addr != NULL);
  } /* showAddrinfo */

#else
#define showAddrinfo(addrinfo_list)
#endif



/**
 *  Read a string, if we do not know how many bytes are available.
 *  This function reads data into a list of buffers. This is done
 *  until enough characters are read or EOF has been reached.
 *  Afterwards the string is allocated, the data is copied from the
 *  buffers and the list of buffers is freed.
 */
static striType receive_and_alloc_stri (socketType inSocket, memSizeType chars_missing,
    errInfoType *err_info)

  {
    struct bufferStruct buffer;
    bufferList currBuffer = &buffer;
    bufferList oldBuffer;
    memSizeType bytes_in_buffer = LIST_BUFFER_SIZE;
    boolType input_ready = TRUE;
    memSizeType result_pos;
    memSizeType result_size = 0;
    striType result;

  /* receive_and_alloc_stri */
    logFunction(printf("receive_and_alloc_stri(%d, " FMT_U_MEM ", *)\n",
                       inSocket, chars_missing););
    buffer.next = NULL;
    while (chars_missing - result_size >= LIST_BUFFER_SIZE &&
           bytes_in_buffer == LIST_BUFFER_SIZE &&
           input_ready) {
      bytes_in_buffer = (memSizeType) recv((os_socketType) inSocket,
                                           cast_send_recv_data(currBuffer->buffer),
                                           cast_buffer_len(LIST_BUFFER_SIZE), 0);
      logMessage(printf("receive_and_alloc_stri: bytes_in_buffer=" FMT_U_MEM "\n",
                        bytes_in_buffer););
      if (unlikely(bytes_in_buffer == (memSizeType) -1)) {
        logError(printf("receive_and_alloc_stri: "
                        "recv(%d, *, " FMT_U_MEM ", 0) failed:\n"
                        "%s=%d\nerror: %s\n",
                        inSocket, (memSizeType) LIST_BUFFER_SIZE,
                        ERROR_INFORMATION););
        bytes_in_buffer = 0;
      } /* if */
      result_size += bytes_in_buffer;
      if (chars_missing > result_size && bytes_in_buffer == LIST_BUFFER_SIZE) {
        currBuffer->next = (bufferList) malloc(sizeof(struct bufferStruct));
        if (unlikely(currBuffer->next == NULL)) {
          logError(printf("receive_and_alloc_stri(%d, " FMT_U_MEM ", *): "
                          "malloc(" FMT_U_MEM ") failed.\n",
                          inSocket, chars_missing, sizeof(struct bufferStruct)););
          *err_info = MEMORY_ERROR;
          result = NULL;
          /* Leave the while loop by setting bytes_in_buffer to zero. */
          bytes_in_buffer = 0;
        } else {
          currBuffer = currBuffer->next;
          currBuffer->next = NULL;
          input_ready = socInputReady(inSocket, 0, 0);
        } /* if */
      } /* if */
    } /* while */
    if (chars_missing > result_size &&
        bytes_in_buffer == LIST_BUFFER_SIZE &&
        input_ready) {
      bytes_in_buffer = (memSizeType) recv((os_socketType) inSocket,
                                           cast_send_recv_data(currBuffer->buffer),
                                           cast_buffer_len(chars_missing - result_size), 0);
      logMessage(printf("receive_and_alloc_stri: bytes_in_buffer=" FMT_U_MEM "\n",
                        bytes_in_buffer););
      if (unlikely(bytes_in_buffer == (memSizeType) -1)) {
        logError(printf("receive_and_alloc_stri: "
                        "recv(%d, *, " FMT_U_MEM ", 0) failed:\n"
                        "%s=%d\nerror: %s\n",
                        inSocket, chars_missing - result_size,
                        ERROR_INFORMATION););
        bytes_in_buffer = 0;
      } /* if */
      result_size += bytes_in_buffer;
    } /* if */
    if (likely(*err_info == OKAY_NO_ERROR)) {
      if (unlikely(!ALLOC_STRI_SIZE_OK(result, result_size))) {
        logError(printf("receive_and_alloc_stri(%d, " FMT_U_MEM ", *): "
                        "ALLOC_STRI_SIZE_OK(*, " FMT_U_MEM ") failed.\n",
                        inSocket, chars_missing, result_size););
        *err_info = MEMORY_ERROR;
      } else {
        result->size = result_size;
        currBuffer = &buffer;
        result_pos = 0;
        while (result_size - result_pos >= LIST_BUFFER_SIZE) {
          memcpy_to_strelem(&result->mem[result_pos], currBuffer->buffer, LIST_BUFFER_SIZE);
          currBuffer = currBuffer->next;
          result_pos += LIST_BUFFER_SIZE;
        } /* while */
        memcpy_to_strelem(&result->mem[result_pos], currBuffer->buffer,
                          result_size - result_pos);
      } /* if */
    } /* if */
    currBuffer = buffer.next;
    while (currBuffer != NULL) {
      oldBuffer = currBuffer;
      currBuffer = currBuffer->next;
      free(oldBuffer);
    } /* while */
    logFunction(printf("receive_and_alloc_stri(%d, " FMT_U_MEM ", %d) --> \"%s\"\n",
                       inSocket, chars_missing, *err_info, striAsUnquotedCStri(result)););
    return result;
  } /* receive_and_alloc_stri */



/**
 *  Create a new accepted connection socket for 'listenerSocket'.
 *  The function waits until at least one connection request is
 *  in the sockets queue of pending connections. Then it extracts
 *  the first connection request from the sockets queue. This
 *  request is accepted and a connection socket is created for it.
 *  @return the accepted connection socket.
 *  @exception FILE_ERROR A system function returns an error.
 *  @exception MEMORY_ERROR An out of memory situation occurred.
 */
socketType socAccept (socketType listenerSocket, bstriType *address)

  {
    memSizeType old_address_size;
    bstriType resized_address;
    sockLenType addrlen;
    os_socketType result;

  /* socAccept */
    logFunction(printf("socAccept(%d, \"%s\")\n",
                       listenerSocket, socAddressCStri(*address)););
    old_address_size = (*address)->size;
    REALLOC_BSTRI_SIZE_OK(resized_address, *address, old_address_size, MAX_ADDRESS_SIZE);
    if (unlikely(resized_address == NULL)) {
      raise_error(MEMORY_ERROR);
      result = (os_socketType) -1;
    } else {
      *address = resized_address;
      COUNT3_BSTRI(old_address_size, MAX_ADDRESS_SIZE);
      addrlen = MAX_ADDRESS_SIZE;
      result = (os_socketType) accept((os_socketType) listenerSocket,
                                      (struct sockaddr *) (*address)->mem, &addrlen);
      if (unlikely(result == INVALID_SOCKET ||
                   ADDRLEN_NEGATIVE(addrlen) ||
                   addrlen > MAX_ADDRESS_SIZE)) {
        /* printf("socAccept(%d) %s=%d %s\n", listenerSocket, ERROR_INFORMATION); */
        REALLOC_BSTRI_SIZE_OK(resized_address, *address, MAX_ADDRESS_SIZE, old_address_size);
        if (resized_address == NULL) {
          (*address)->size = MAX_ADDRESS_SIZE;
        } else {
          *address = resized_address;
          COUNT3_BSTRI(MAX_ADDRESS_SIZE, old_address_size);
        } /* if */
        logError(printf("socAccept: accept(%d, \"%s\") failed:\n"
                        "%s=%d\nerror: %s\n",
                        listenerSocket, socAddressCStri(*address),
                        ERROR_INFORMATION););
        raise_error(FILE_ERROR);
      } else {
        REALLOC_BSTRI_SIZE_OK(resized_address, *address, MAX_ADDRESS_SIZE,
                              (memSizeType) addrlen);
        if (unlikely(resized_address == NULL)) {
          (*address)->size = MAX_ADDRESS_SIZE;
          raise_error(MEMORY_ERROR);
          result = (os_socketType) -1;
        } else {
          *address = resized_address;
          COUNT3_BSTRI(MAX_ADDRESS_SIZE, (memSizeType) addrlen);
          (*address)->size = (memSizeType) addrlen;
        } /* if */
      } /* if */
    } /* if */
    logFunction(printf("socAccept(%d, \"%s\") --> %d\n",
                       listenerSocket, socAddressCStri(*address), result););
    return (socketType) result;
  } /* socAccept */



intType socAddrFamily (const const_bstriType address)

  {
    const struct sockaddr *addr;
    intType result;

  /* socAddrFamily */
    logFunction(printf("socAddrFamily(\"%s\")\n", socAddressCStri(address)););
    if (unlikely(address->size < sizeof(struct sockaddr))) {
      logError(printf("socAddrFamily(\"%s\"): Size of address too small.\n",
                      socAddressCStri(address)););
      raise_error(RANGE_ERROR);
      result = 0;
    } else {
      addr = (const struct sockaddr *) address->mem;
      result = addr->sa_family;
      /* printf("socAddrFamily --> %d\n", result); */
    } /* if */
    logFunction(printf("socAddrFamily(\"%s\") --> " FMT_D "\n",
                       socAddressCStri(address), result););
    return result;
  } /* socAddrFamily */



/**
 *  Get the numeric (IP) address of the host at 'address'.
 *  IPv4 addresses return the socketAddress in dot notation (e.g.:
 *  "192.0.2.235") and IPv6 addresses return the socketAddress in
 *  colon notation (e.g.: "fe80:0:0:0:202:b3ff:fe1e:8329").
 *  @return the IP address of the specified host.
 *  @exception RANGE_ERROR The address is neither AF_INET nor AF_INET6.
 */
striType socAddrNumeric (const const_bstriType address)

  {
    const struct sockaddr *addr;
    char buffer[MAX_SOCK_ADDRESS_LEN + NULL_TERMINATION_LEN];
    striType result;

  /* socAddrNumeric */
    logFunction(printf("socAddrNumeric(\"%s\")\n", socAddressCStri(address)););
    if (unlikely(address->size < sizeof(struct sockaddr))) {
      logError(printf("socAddrNumeric(\"%s\"): Size of address too small.\n",
                      socAddressCStri(address)););
      raise_error(RANGE_ERROR);
      result = NULL;
    } else {
      addr = (const struct sockaddr *) address->mem;
      switch (addr->sa_family) {
        case AF_INET:
          if (unlikely(address->size != sizeof(struct sockaddr_in))) {
            logError(printf("socAddrNumeric(\"%s\"): Size of address wrong for AF_INET.\n",
                            socAddressCStri(address)););
            raise_error(RANGE_ERROR);
            result = NULL;
          } else {
            const struct sockaddr_in *inet_address = (const struct sockaddr_in *) address->mem;
            uint32Type ip4_address = ntohl(inet_address->sin_addr.s_addr);

            sprintf(buffer, "%d.%d.%d.%d",
                (ip4_address >> 24) & 255,
                (ip4_address >> 16) & 255,
                (ip4_address >>  8) & 255,
                 ip4_address        & 255);
            result = cstri_to_stri(buffer);
          } /* if */
          break;
#if HAS_GETADDRINFO || defined INET6_SERVER_ADDRESS
        case AF_INET6:
          if (unlikely(address->size != sizeof(struct sockaddr_in6))) {
            logError(printf("socAddrNumeric(\"%s\"): Size of address wrong for AF_INET6.\n",
                            socAddressCStri(address)););
            raise_error(RANGE_ERROR);
            result = NULL;
          } else {
            const struct sockaddr_in6 *inet6_address =
                (const struct sockaddr_in6 *) address->mem;
            unsigned int digitGroup[8];
            int pos;

            for (pos = 0; pos <= 7; pos++) {
              digitGroup[pos] =
                  (unsigned int) (inet6_address->sin6_addr.s6_addr[pos << 1]) << 8 |
                  (unsigned int) (inet6_address->sin6_addr.s6_addr[(pos << 1) + 1]);
            } /* for */
            sprintf(buffer, "%x:%x:%x:%x:%x:%x:%x:%x",
                digitGroup[0], digitGroup[1], digitGroup[2], digitGroup[3],
                digitGroup[4], digitGroup[5], digitGroup[6], digitGroup[7]);
            result = cstri_to_stri(buffer);
          } /* if */
          break;
#endif
        default:
          logError(printf("socAddrNumeric(\"%s\"): Address neither AF_INET nor AF_INET6.\n",
                          socAddressCStri(address)););
          raise_error(RANGE_ERROR);
          result = NULL;
          break;
      } /* switch */
    } /* if */
    logFunction(printf("socAddrNumeric(\"%s\") --> \"%s\"\n",
                       socAddressCStri(address), striAsUnquotedCStri(result)););
    return result;
  } /* socAddrNumeric */



striType socAddrService (const const_bstriType address)

  {
    const struct sockaddr *addr;
    striType result;

  /* socAddrService */
    logFunction(printf("socAddrService(\"%s\")\n", socAddressCStri(address)););
    if (unlikely(address->size < sizeof(struct sockaddr))) {
      logError(printf("socAddrService(\"%s\"): Size of address too small.\n",
                      socAddressCStri(address)););
      raise_error(RANGE_ERROR);
      result = NULL;
    } else {
      addr = (const struct sockaddr *) address->mem;
      switch (addr->sa_family) {
        case AF_INET:
          if (unlikely(address->size != sizeof(struct sockaddr_in))) {
            logError(printf("socAddrService(\"%s\"): Size of address wrong for AF_INET.\n",
                            socAddressCStri(address)););
            raise_error(RANGE_ERROR);
            result = NULL;
          } else {
            intType port;
            const struct sockaddr_in *inet_address;
            inet_address = (const struct sockaddr_in *) address->mem;
            port = ntohs(inet_address->sin_port);       /* short, network byte order */
            result = intStr(port);
          } /* if */
          break;
#if HAS_GETADDRINFO || defined INET6_SERVER_ADDRESS
        case AF_INET6:
          if (unlikely(address->size != sizeof(struct sockaddr_in6))) {
            logError(printf("socAddrService(\"%s\"): Size of address wrong for AF_INET6.\n",
                            socAddressCStri(address)););
            raise_error(RANGE_ERROR);
            result = NULL;
          } else {
            intType port;
            const struct sockaddr_in6 *inet6_address;
            inet6_address = (const struct sockaddr_in6 *) address->mem;
            port = ntohs(inet6_address->sin6_port);     /* short, network byte order */
            result = intStr(port);
          } /* if */
          break;
#endif
        default:
          logError(printf("socAddrService(\"%s\"): Address neither AF_INET nor AF_INET6.\n",
                          socAddressCStri(address)););
          raise_error(RANGE_ERROR);
          result = NULL;
          break;
      } /* switch */
    } /* if */
    logFunction(printf("socAddrService(\"%s\") --> \"%s\"\n",
                       socAddressCStri(address), striAsUnquotedCStri(result)););
    return result;
  } /* socAddrService */



/**
 *  Assign the specified 'address' to the 'listenerSocket'.
 *  @param address An internet listener socket address.
 *  @exception FILE_ERROR A system function returns an error.
 */
void socBind (socketType listenerSocket, const_bstriType address)

  { /* socBind */
    logFunction(printf("socBind(%d, \"%s\")\n",
                       listenerSocket, socAddressCStri(address)););
    if (unlikely(bind((os_socketType) listenerSocket,
                      (const struct sockaddr *) address->mem,
                      (sockLenType) address->size) != 0)) {
      logError(printf("socBind: bind(%d, \"%s\") failed:\n"
                      "%s=%d\nerror: %s\n",
                      listenerSocket, socAddressCStri(address),
                      ERROR_INFORMATION););
      raise_error(FILE_ERROR);
    } /* if */
  } /* socBind */


/**
 *  Close the socket 'aSocket'.
 *  @exception FILE_ERROR A system function returns an error.
 */
void socClose (socketType aSocket)

  {
    int close_result;

  /* socClose */
    logFunction(printf("socClose(%d)\n", aSocket););
    shutdown((os_socketType) aSocket, SHUT_RDWR);
#if SOCKET_LIB == UNIX_SOCKETS
    close_result = close((os_socketType) aSocket);
#elif SOCKET_LIB == WINSOCK_SOCKETS
    close_result = closesocket((os_socketType) aSocket);
#endif
    if (unlikely(close_result != 0)) {
      logError(printf("socClose: close(%d) failed:\n"
                      "%s=%d\nerror: %s\n",
                      aSocket, ERROR_INFORMATION););
      raise_error(FILE_ERROR);
    } /* if */
  } /* socClose */



/**
 *  Connect 'aSocket' to the given 'address'.
 *  @exception FILE_ERROR A system function returns an error.
 */
void socConnect (socketType aSocket, const_bstriType address)

  { /* socConnect */
    logFunction(printf("socConnect(%d, \"%s\")\n",
                       aSocket, socAddressCStri(address)););
    if (unlikely(connect((os_socketType) aSocket,
                         (const struct sockaddr *) address->mem,
                         (sockLenType) address->size) != 0)) {
      logError(printf("socConnect: connect(%d, \"%s\") failed:\n"
                      "%s=%d\nerror: %s\n",
                      aSocket, socAddressCStri(address),
                      ERROR_INFORMATION););
      raise_error(FILE_ERROR);
    } /* if */
  } /* socConnect */



/**
 *  Read a character from 'inSocket'.
 *  @return the character read.
 */
charType socGetc (socketType inSocket, charType *const eofIndicator)

  {
    unsigned char ch;
    memSizeType bytes_received;
    charType result;

  /* socGetc */
    logFunction(printf("socGetc(%d, '\\" FMT_U32 ";')\n",
                       inSocket, *eofIndicator););
    bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                        cast_send_recv_data(&ch), 1, 0);
    logMessage(printf("socGetc: bytes_received=" FMT_U_MEM "\n",
                      bytes_received););
    if (bytes_received != 1) {
      if (unlikely(inSocket == (socketType) -1)) {
        logError(printf("socGetc(%d, '\\" FMT_U32 ";'): Invalid socket.\n",
                        inSocket, *eofIndicator););
        raise_error(FILE_ERROR);
        result = (charType) EOF;
      } else {
        *eofIndicator = (charType) EOF;
        result = (charType) EOF;
      } /* if */
    } else {
      result = (charType) ch;
    } /* if */
    logFunction(printf("socGetc (%d, '\\" FMT_U32 ";') --> '\\" FMT_U32 ";'\n",
                       inSocket, *eofIndicator, result););
    return result;
  } /* socGetc */



/**
 *  Read a string with a maximum length from 'inSocket'.
 *  @return the string read.
 *  @exception RANGE_ERROR The length is negative.
 *  @exception MEMORY_ERROR Not enough memory to represent the result.
 */
striType socGets (socketType inSocket, intType length, charType *const eofIndicator)

  {
    memSizeType chars_requested;
    memSizeType result_size;
    errInfoType err_info = OKAY_NO_ERROR;
    striType resized_result;
    striType result;

  /* socGets */
    logFunction(printf("socGets(%d, " FMT_D ", '\\" FMT_U32 ";')\n",
                       inSocket, length, *eofIndicator););
    if (unlikely(inSocket == (socketType) -1)) {
        logError(printf("socGets(%d, " FMT_D ", '\\" FMT_U32 ";'): "
                        "Invalid socket.\n",
                        inSocket, length, *eofIndicator););
      raise_error(FILE_ERROR);
      result = NULL;
    } else if (unlikely(length <= 0)) {
      if (unlikely(length != 0)) {
        logError(printf("socGets(%d, " FMT_D ", *): Negative length.\n",
                        inSocket, length););
        raise_error(RANGE_ERROR);
        result = NULL;
      } else {
        if (unlikely(!ALLOC_STRI_SIZE_OK(result, 0))) {
          raise_error(MEMORY_ERROR);
        } else {
          result->size = 0;
        } /* if */
      } /* if */
    } else {
      if ((uintType) length > MAX_MEMSIZETYPE) {
        chars_requested = MAX_MEMSIZETYPE;
      } else {
        chars_requested = (memSizeType) length;
      } /* if */
      if (chars_requested <= BUFFER_SIZE) {
        ucharType buffer[BUFFER_SIZE];

        result_size = (memSizeType) recv((os_socketType) inSocket,
                                         cast_send_recv_data(buffer),
                                         cast_buffer_len(chars_requested), 0);
        logMessage(printf("socGets: result_size=" FMT_U_MEM "\n",
                          result_size););
        if (unlikely(result_size == (memSizeType) -1)) {
          logError(printf("socGets: "
                          "recv(%d, *, " FMT_U_MEM ", 0) failed:\n"
                          "%s=%d\nerror: %s\n",
                          inSocket, chars_requested,
                          ERROR_INFORMATION););
          result_size = 0;
        } /* if */
        if (unlikely(!ALLOC_STRI_CHECK_SIZE(result, result_size))) {
          logError(printf("socGets(%d, " FMT_D ", *): "
                          "Out of memory when allocating result.\n",
                          inSocket, length););
          raise_error(MEMORY_ERROR);
          result = NULL;
        } else {
          memcpy_to_strelem(result->mem, buffer, result_size);
          result->size = result_size;
          if (result_size == 0) {
            *eofIndicator = (charType) EOF;
          } /* if */
        } /* if */
      } else {
        if (chars_requested > GETS_DEFAULT_SIZE) {
          /* Read a string, if we do not know how many bytes are available. */
          result = receive_and_alloc_stri(inSocket, chars_requested, &err_info);
          if (unlikely(err_info != OKAY_NO_ERROR)) {
            raise_error(err_info);
          } else if (result->size == 0) {
            *eofIndicator = (charType) EOF;
          } /* if */
        } else {
          if (unlikely(!ALLOC_STRI_CHECK_SIZE(result, chars_requested))) {
            logError(printf("socGets(%d, " FMT_D ", *): "
                            "Out of memory when allocating result.\n",
                            inSocket, length););
            raise_error(MEMORY_ERROR);
            result = NULL;
          } else {
            result_size = (memSizeType) recv((os_socketType) inSocket,
                                             cast_send_recv_data(result->mem),
                                             cast_buffer_len(chars_requested), 0);
            logMessage(printf("socGets: result_size=" FMT_U_MEM "\n",
                              result_size););
            if (unlikely(result_size == (memSizeType) -1)) {
              logError(printf("socGets: "
                              "recv(%d, *, " FMT_U_MEM ", 0) failed:\n"
                              "%s=%d\nerror: %s\n",
                              inSocket, chars_requested,
                              ERROR_INFORMATION););
              result_size = 0;
            } /* if */
            memcpy_to_strelem(result->mem, (ustriType) result->mem, result_size);
            result->size = result_size;
            if (result_size < chars_requested) {
              if (result_size == 0) {
                *eofIndicator = (charType) EOF;
              } /* if */
              REALLOC_STRI_SIZE_SMALLER2(resized_result, result, chars_requested, result_size);
              if (unlikely(resized_result == NULL)) {
                FREE_STRI2(result, chars_requested);
                logError(printf("socGets(%d, " FMT_D ", *): "
                                "Out of memory when allocating result.\n",
                                inSocket, length););
                raise_error(MEMORY_ERROR);
                result = NULL;
              } else {
                result = resized_result;
              } /* if */
            } /* if */
          } /* if */
        } /* if */
      } /* if */
    } /* if */
    logFunction(printf("socGets(%d, " FMT_D ", '\\" FMT_U32 ";') --> \"%s\"\n",
                       inSocket, length, *eofIndicator, striAsUnquotedCStri(result)););
    return result;
  } /* socGets */



/**
 *  Determine the hostname.
 *  @return the hostname.
 *  @exception MEMORY_ERROR Not enough memory to represent the result.
 */
striType socGetHostname (void)

  {
    char name[MAX_HOSTNAME_LENGTH];
    striType result;

  /* socGetHostname */
    check_initialization(NULL);
    if (unlikely(gethostname(name, MAX_HOSTNAME_LENGTH) != 0)) {
      raise_error(MEMORY_ERROR);
      result = NULL;
    } else {
      result = cstri8_or_cstri_to_stri(name);
    } /* if */
    return result;
  } /* socGetHostname */



/**
 *  Get the local address of the socket 'sock'.
 *  @return the address to which the socket 'sock' is bound.
 *  @exception FILE_ERROR A system function returns an error.
 *  @exception MEMORY_ERROR Not enough memory to represent the result.
 */
bstriType socGetLocalAddr (socketType sock)

  {
    sockLenType addrlen;
    int getsockname_result;
    bstriType resized_address;
    bstriType address;

  /* socGetLocalAddr */
    logFunction(printf("socGetLocalAddr(%d)\n", sock););
    if (unlikely(!ALLOC_BSTRI_SIZE_OK(address, MAX_ADDRESS_SIZE))) {
      raise_error(MEMORY_ERROR);
    } else {
      addrlen = MAX_ADDRESS_SIZE;
      getsockname_result = getsockname((os_socketType) sock,
                                       (struct sockaddr *) address->mem, &addrlen);
      if (unlikely(getsockname_result != 0 ||
                   ADDRLEN_NEGATIVE(addrlen) ||
                   addrlen > MAX_ADDRESS_SIZE)) {
        FREE_BSTRI(address, MAX_ADDRESS_SIZE);
        logError(printf("socGetLocalAddr: getsockname(%d, ...) failed:\n"
                        "%s=%d\nerror: %s\n",
                        sock, ERROR_INFORMATION););
        raise_error(FILE_ERROR);
        address = NULL;
      } else {
        REALLOC_BSTRI_SIZE_OK(resized_address, address, MAX_ADDRESS_SIZE,
                              (memSizeType) addrlen);
        if (unlikely(resized_address == NULL)) {
          FREE_BSTRI(address, MAX_ADDRESS_SIZE);
          raise_error(MEMORY_ERROR);
          address = NULL;
        } else {
          address = resized_address;
          COUNT3_BSTRI(MAX_ADDRESS_SIZE, (memSizeType) addrlen);
          address->size = (memSizeType) addrlen;
        } /* if */
      } /* if */
    } /* if */
    logFunction(printf("socGetLocalAddr(%d) --> \"%s\"\n",
                       sock, socAddressCStri(address)););
    return address;
  } /* socGetLocalAddr */



/**
 *  Get the address of the peer to which the socket 'sock' is connected.
 *  @return the address of the peer connected to the socket 'sock'.
 *  @exception FILE_ERROR A system function returns an error.
 *  @exception MEMORY_ERROR Not enough memory to represent the result.
 */
bstriType socGetPeerAddr (socketType sock)

  {
    sockLenType addrlen;
    int getpeername_result;
    bstriType resized_address;
    bstriType address;

  /* socGetPeerAddr */
    logFunction(printf("socGetPeerAddr(%d)\n", sock););
    if (unlikely(!ALLOC_BSTRI_SIZE_OK(address, MAX_ADDRESS_SIZE))) {
      raise_error(MEMORY_ERROR);
    } else {
      addrlen = MAX_ADDRESS_SIZE;
      getpeername_result = getpeername((os_socketType) sock,
                                       (struct sockaddr *) address->mem, &addrlen);
      if (unlikely(getpeername_result != 0 ||
                   ADDRLEN_NEGATIVE(addrlen) ||
                   addrlen > MAX_ADDRESS_SIZE)) {
        FREE_BSTRI(address, MAX_ADDRESS_SIZE);
        logError(printf("socGetPeerAddr: getpeername(%d, ...) failed:\n"
                        "%s=%d\nerror: %s\n",
                        sock, ERROR_INFORMATION););
        raise_error(FILE_ERROR);
        address = NULL;
      } else {
        REALLOC_BSTRI_SIZE_OK(resized_address, address, MAX_ADDRESS_SIZE,
                              (memSizeType) addrlen);
        if (unlikely(resized_address == NULL)) {
          FREE_BSTRI(address, MAX_ADDRESS_SIZE);
          raise_error(MEMORY_ERROR);
          address = NULL;
        } else {
          address = resized_address;
          COUNT3_BSTRI(MAX_ADDRESS_SIZE, (memSizeType) addrlen);
          address->size = (memSizeType) addrlen;
        } /* if */
      } /* if */
    } /* if */
    logFunction(printf("socGetPeerAddr(%d) --> \"%s\"\n",
                       sock, socAddressCStri(address)););
    return address;
  } /* socGetPeerAddr */



/**
 *  Determine if at least one character can be read successfully.
 *  This function allows a socket to be handled like an iterator.
 *  Since socHasNext peeks the next character from the socket
 *  it may block.
 *  @return FALSE if socGetc would return EOF, TRUE otherwise.
 */
boolType socHasNext (socketType inSocket)

  {
    unsigned char next_char;
    memSizeType bytes_received;
    boolType hasNext;

  /* socHasNext */
    logFunction(printf("socHasNext(%d)\n", inSocket););
    bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                        cast_send_recv_data(&next_char), 1, MSG_PEEK);
    if (bytes_received != 1) {
      logMessage(printf("socHasNext: bytes_received=" FMT_U_MEM "\n",
                        bytes_received););
      if (unlikely(inSocket == (socketType) -1)) {
        logError(printf("socHasNext(%d): Invalid socket.\n", inSocket););
        raise_error(FILE_ERROR);
      } /* if */
      hasNext = FALSE;
    } else {
      logMessage(printf("socHasNext: next_char=%d\n", next_char););
      hasNext = TRUE;
    } /* if */
    logFunction(printf("socHasNext(%d) --> %d\n", inSocket, hasNext););
    return hasNext;
  } /* socHasNext */



/**
 *  Create an internet socket address of a port at a host.
 *  The 'hostName' is either a host name (e.g.: "www.example.org"),
 *  or an IPv4 address in standard dot notation (e.g.: "192.0.2.235").
 *  Operating systems supporting IPv6 may also accept an IPv6 address
 *  in colon notation.
 *  @return the internet socket address, or
 *          an empty bstring if the host cannot be found.
 *  @exception FILE_ERROR A system function returns an error.
 *  @exception RANGE_ERROR The port is not in the range 0 to 65535 or
 *             hostName cannot be converted to the system string type.
 *  @exception MEMORY_ERROR Not enough memory to convert 'hostName'.
 *             to the system representation or not enough memory to
 *             represent the result.
 */
bstriType socInetAddr (const const_striType hostName, intType port)

  {
    cstriType os_hostName;
#if HAS_GETADDRINFO
    char serviceName[UINT16TYPE_DECIMAL_SIZE + NULL_TERMINATION_LEN];
    struct addrinfo *addrinfo_list;
    struct addrinfo *result_addrinfo;
    struct addrinfo hints;
    int getaddrinfo_result;
#else
    struct hostent *host_ent;
    struct sockaddr_in *inet_address;
#endif
    errInfoType err_info = OKAY_NO_ERROR;
    bstriType result;

  /* socInetAddr */
    logFunction(printf("socInetAddr(\"%s\", " FMT_D ")\n",
                       striAsUnquotedCStri(hostName), port););
    check_initialization(NULL);
    if (unlikely(port < 0 || port > 65535)) {
      logError(printf("socInetAddr(\"%s\", " FMT_D "): "
                      "Port not in allowed range.\n",
                      striAsUnquotedCStri(hostName), port););
      raise_error(RANGE_ERROR);
      result = NULL;
    } else {
      os_hostName = stri_to_cstri8(hostName, &err_info);
      if (unlikely(os_hostName == NULL)) {
        logError(printf("socInetAddr: stri_to_cstri8(\"%s\", *) failed:\n"
                        "err_info=%d\n",
                        striAsUnquotedCStri(hostName), err_info););
        raise_error(err_info);
        result = NULL;
      } else {
#if HAS_GETADDRINFO
        sprintf(serviceName, FMT_U16, (uint16Type) port);
        memset(&hints, 0, sizeof(struct addrinfo));
        hints.ai_family = AF_UNSPEC;
        hints.ai_socktype = SOCK_STREAM;
        getaddrinfo_result = getaddrinfo(os_hostName, serviceName, &hints, &addrinfo_list);
        if (unlikely(getaddrinfo_result == EAI_NONAME)) {
          /* Check if the host name is the name of the local host */
          char localHostName[MAX_HOSTNAME_LENGTH];
          if (gethostname(localHostName, MAX_HOSTNAME_LENGTH) == 0 &&
              strcmp(os_hostName, localHostName) == 0) {
            getaddrinfo_result = getaddrinfo("localhost", serviceName, &hints,
                                             &addrinfo_list);
          } /* if */
        } /* if */
        if (unlikely(getaddrinfo_result != 0)) {
          logMessage(printf("getaddrinfo(\"%s\", \"%s\") -> %d\n",
                            os_hostName, serviceName, getaddrinfo_result));
          if (getaddrinfo_result == EAI_NONAME || getaddrinfo_result == EAI_AGAIN
#ifdef EAI_NODATA
              || getaddrinfo_result == EAI_NODATA
#endif
          ) {
            free_cstri8(os_hostName, hostName);
            /* Return empty address */
            if (unlikely(!ALLOC_BSTRI_SIZE_OK(result, 0))) {
              raise_error(MEMORY_ERROR);
            } else {
              result->size = 0;
            } /* if */
          } else {
            logError(printf("socInetAddr(\"%s\", " FMT_D "): "
                            "getaddrinfo(\"%s\", \"%s\", *, *) failed with %d:\n"
                            "strerror: %s\n"
                            "%s=%d\nerror: %s\n",
                            striAsUnquotedCStri(hostName), port,
                            os_hostName, serviceName, getaddrinfo_result,
                            gai_strerror(getaddrinfo_result),
                            ERROR_INFORMATION););
            /*
            printf("EAI_AGAIN=%d  EAI_BADFLAGS=%d  EAI_FAIL=%d  EAI_FAMILY=%d\n",
                EAI_AGAIN, EAI_BADFLAGS, EAI_FAIL, EAI_FAMILY);
            printf("EAI_MEMORY=%d  EAI_NONAME=%d  EAI_SERVICE=%d  EAI_SOCKTYPE=%d\n",
                EAI_MEMORY, EAI_NONAME, EAI_SERVICE, EAI_SOCKTYPE);
            */
            /* printf("EAI_SYSTEM=%d  EAI_OVERFLOW=%d\n",
                EAI_SYSTEM, EAI_OVERFLOW); */
            /* printf("EAI_ADDRFAMILY=%d  EAI_NODATA=%d\n",
                EAI_ADDRFAMILY, EAI_NODATA); */
            free_cstri8(os_hostName, hostName);
            raise_error(FILE_ERROR);
            result = NULL;
          } /* if */
        } else {
          showAddrinfo(addrinfo_list);
          result_addrinfo = select_addrinfo(addrinfo_list, AF_INET, AF_INET6);
          if (unlikely(ADDRLEN_NEGATIVE(result_addrinfo->ai_addrlen) ||
                       !ALLOC_BSTRI_SIZE_OK(result,
                           (memSizeType) result_addrinfo->ai_addrlen))) {
            free_cstri8(os_hostName, hostName);
            freeaddrinfo(addrinfo_list);
            raise_error(MEMORY_ERROR);
            result = NULL;
          } else {
            result->size = (memSizeType) result_addrinfo->ai_addrlen;
            memcpy(result->mem, result_addrinfo->ai_addr,
                   (memSizeType) result_addrinfo->ai_addrlen);
            free_cstri8(os_hostName, hostName);
            freeaddrinfo(addrinfo_list);
          } /* if */
        } /* if */
#else
        host_ent = gethostbyname(os_hostName);
        if (host_ent == NULL && h_errno == TRY_AGAIN) {
          /*
          printf("***** h_errno=%d\n", h_errno);
          printf("***** os_hostName=\"%s\"\n", os_hostName);
          printf("***** port=" FMT_D "\n", port);
          printf("***** hostName=%s\n", striAsUnquotedCStri(hostName));
          */
          host_ent = gethostbyname(os_hostName);
        } /* if */
        if (unlikely(host_ent == NULL)) {
          /* printf("***** gethostbyname(\"%s\"): h_errno=%d\n", os_hostName, h_errno);
             printf("HOST_NOT_FOUND=%d  NO_DATA=%d  NO_RECOVERY=%d  TRY_AGAIN=%d\n",
                 HOST_NOT_FOUND, NO_DATA, NO_RECOVERY, TRY_AGAIN); */
          free_cstri8(os_hostName, hostName);
          /* Return empty address */
          if (unlikely(!ALLOC_BSTRI_SIZE_OK(result, 0))) {
            raise_error(MEMORY_ERROR);
          } else {
            result->size = 0;
          } /* if */
        } else {
          /*
          printf("Host name:      %s\n", host_ent->h_name);
          printf("Port:           " FMT_D "\n", port);
          printf("Address type:   %d\n", host_ent->h_addrtype);
          printf("Address type:   %d\n", AF_INET);
          printf("Address length: %d\n", host_ent->h_length);
          printf("Address length: %d\n", sizeof(struct sockaddr_in));
          printf("IP Address:     %s\n", inet_ntoa(*((struct in_addr *)host_ent->h_addr)));
          */
          free_cstri8(os_hostName, hostName);
          if (host_ent->h_addrtype == AF_INET &&
              host_ent->h_length == sizeof(inet_address->sin_addr.s_addr)) {
            if (unlikely(!ALLOC_BSTRI_SIZE_OK(result, sizeof(struct sockaddr_in)))) {
              raise_error(MEMORY_ERROR);
            } else {
              result->size = sizeof(struct sockaddr_in);
              inet_address = (struct sockaddr_in *) result->mem;
              inet_address->sin_family = host_ent->h_addrtype;
              inet_address->sin_port = htons((uint16Type) port); /* short, network byte order */
              memcpy(&inet_address->sin_addr.s_addr, host_ent->h_addr,
                     (size_t) host_ent->h_length);
              memset(inet_address->sin_zero, '\0', sizeof(inet_address->sin_zero));
              /* {
                uint32Type ip4_address = ntohl(inet_address->sin_addr.s_addr);
                printf("ip4_address=%d.%d.%d.%d\n",
                    (ip4_address >> 24  & 255,
                    (ip4_address >> 16) & 255,
                    (ip4_address >>  8) & 255,
                     ip4_address        & 255);
              } */
            } /* if */
          } else {
            /* printf("socInetAddr: addrtype=%d\n", host_ent->h_addrtype); */
            /* raise_error(FILE_ERROR);
               result = NULL; */
            /* Return empty address */
            if (unlikely(!ALLOC_BSTRI_SIZE_OK(result, 0))) {
              raise_error(MEMORY_ERROR);
            } else {
              result->size = 0;
            } /* if */
          } /* if */
        } /* if */
#endif
      } /* if */
    } /* if */
    logFunction(printf("socInetAddr(\"%s\", " FMT_D ") --> \"%s\"\n",
                       striAsUnquotedCStri(hostName), port,
                       socAddressCStri(result)););
    return result;
  } /* socInetAddr */



/**
 *  Create an internet socket address of a port at localhost.
 *  @return the internet socket address.
 *  @exception FILE_ERROR A system function returns an error.
 *  @exception RANGE_ERROR The port is not in the range 0 to 65535.
 *  @exception MEMORY_ERROR Not enough memory to represent the result.
 */
bstriType socInetLocalAddr (intType port)

  {
#if HAS_GETADDRINFO
    char serviceName[UINT16TYPE_DECIMAL_SIZE + NULL_TERMINATION_LEN];
    struct addrinfo *addrinfo_list;
    struct addrinfo *result_addrinfo;
    struct addrinfo hints;
    int getaddrinfo_result;
#else
    struct sockaddr_in *inet_address;
#endif
    bstriType result;

  /* socInetLocalAddr */
    logFunction(printf("socInetLocalAddr(" FMT_D ")\n", port););
    check_initialization(NULL);
    if (unlikely(port < 0 || port > 65535)) {
      logError(printf("socInetLocalAddr(" FMT_D "): "
                      "Port not in allowed range.\n",
                      port););
      raise_error(RANGE_ERROR);
      result = NULL;
    } else {
#if HAS_GETADDRINFO
      sprintf(serviceName, FMT_U16, (uint16Type) port);
      memset(&hints, 0, sizeof(struct addrinfo));
      hints.ai_family = AF_UNSPEC;
      hints.ai_socktype = SOCK_STREAM;
      getaddrinfo_result = getaddrinfo(NULL, serviceName, &hints, &addrinfo_list);
      if (unlikely(getaddrinfo_result != 0)) {
        logError(printf("socInetLocalAddr(" FMT_D "): "
                        "getaddrinfo(NULL, %s, *, *) failed with %d:\n"
                        "strerror: %s\n"
                        "%s=%d\nerror: %s\n",
                        port, serviceName, getaddrinfo_result,
                        gai_strerror(getaddrinfo_result),
                        ERROR_INFORMATION););
        raise_error(FILE_ERROR);
        result = NULL;
      } else {
        showAddrinfo(addrinfo_list);
        result_addrinfo = select_addrinfo(addrinfo_list, AF_INET, AF_INET6);
        if (unlikely(ADDRLEN_NEGATIVE(result_addrinfo->ai_addrlen) ||
                     !ALLOC_BSTRI_SIZE_OK(result,
                         (memSizeType) result_addrinfo->ai_addrlen))) {
          freeaddrinfo(addrinfo_list);
          raise_error(MEMORY_ERROR);
          result = NULL;
        } else {
          result->size = (memSizeType) result_addrinfo->ai_addrlen;
          memcpy(result->mem, result_addrinfo->ai_addr,
                 (memSizeType) result_addrinfo->ai_addrlen);
          freeaddrinfo(addrinfo_list);
        } /* if */
      } /* if */
#else
      if (unlikely(!ALLOC_BSTRI_SIZE_OK(result, sizeof(struct sockaddr_in)))) {
        raise_error(MEMORY_ERROR);
      } else {
        result->size = sizeof(struct sockaddr_in);
        inet_address = (struct sockaddr_in *) result->mem;
        inet_address->sin_family = AF_INET;
        inet_address->sin_port = htons((uint16Type) port);      /* short, network byte order */
        inet_address->sin_addr.s_addr = htonl(INADDR_LOOPBACK); /* local host */
        memset(inet_address->sin_zero, '\0', sizeof(inet_address->sin_zero));
      } /* if */
#endif
    } /* if */
    logFunction(printf("socInetLocalAddr(" FMT_D ") --> \"%s\"\n",
                       port, socAddressCStri(result)););
    return result;
  } /* socInetLocalAddr */



/**
 *  Create an internet listener socket address of a port at localhost.
 *  @return the internet listener socket address.
 *  @exception FILE_ERROR A system function returns an error.
 *  @exception RANGE_ERROR The port is not in the range 0 to 65535.
 *  @exception MEMORY_ERROR Not enough memory to represent the result.
 */
bstriType socInetServAddr (intType port)

  {
#if HAS_GETADDRINFO
    char serviceName[UINT16TYPE_DECIMAL_SIZE + NULL_TERMINATION_LEN];
    struct addrinfo *addrinfo_list;
    struct addrinfo *result_addrinfo;
    struct addrinfo hints;
    int getaddrinfo_result;
#else
#ifdef INET6_SERVER_ADDRESS
    struct sockaddr_in6 *inet6_address;
#else
    struct sockaddr_in *inet_address;
#endif
#endif
    bstriType result;

  /* socInetServAddr */
    logFunction(printf("socInetServAddr(" FMT_D ")\n", port););
    check_initialization(NULL);
    if (unlikely(port < 0 || port > 65535)) {
      logError(printf("socInetServAddr(" FMT_D "): "
                      "Port not in allowed range.\n",
                      port););
      raise_error(RANGE_ERROR);
      result = NULL;
    } else {
#if HAS_GETADDRINFO
      sprintf(serviceName, FMT_U16, (uint16Type) port);
      memset(&hints, 0, sizeof(struct addrinfo));
      hints.ai_family = AF_UNSPEC;
      hints.ai_socktype = SOCK_STREAM;
      hints.ai_flags = AI_PASSIVE;
      getaddrinfo_result = getaddrinfo(NULL, serviceName, &hints, &addrinfo_list);
      if (unlikely(getaddrinfo_result != 0)) {
        logError(printf("socInetServAddr(" FMT_D "): "
                        "getaddrinfo(NULL, %s, *, *) failed with %d:\n"
                        "strerror: %s\n"
                        "%s=%d\nerror: %s\n",
                        port, serviceName, getaddrinfo_result,
                        gai_strerror(getaddrinfo_result),
                        ERROR_INFORMATION););
        raise_error(FILE_ERROR);
        result = NULL;
      } else {
        showAddrinfo(addrinfo_list);
        result_addrinfo = select_addrinfo(addrinfo_list, AF_INET, AF_INET6);
        if (unlikely(ADDRLEN_NEGATIVE(result_addrinfo->ai_addrlen) ||
                     !ALLOC_BSTRI_SIZE_OK(result,
                         (memSizeType) result_addrinfo->ai_addrlen))) {
          freeaddrinfo(addrinfo_list);
          raise_error(MEMORY_ERROR);
          result = NULL;
        } else {
          result->size = (memSizeType) result_addrinfo->ai_addrlen;
          memcpy(result->mem, result_addrinfo->ai_addr,
                 (memSizeType) result_addrinfo->ai_addrlen);
          freeaddrinfo(addrinfo_list);
        } /* if */
      } /* if */
#else
#ifdef INET6_SERVER_ADDRESS
      if (unlikely(!ALLOC_BSTRI_SIZE_OK(result, sizeof(struct sockaddr_in6)))) {
        raise_error(MEMORY_ERROR);
      } else {
        result->size = sizeof(struct sockaddr_in6);
        inet6_address = (struct sockaddr_in6 *) result->mem;
        inet6_address->sin6_family = AF_INET6;
        inet6_address->sin6_port = htons((uint16Type) port); /* short, network byte order */
        inet6_address->sin6_flowinfo = 0;
        memcpy(&inet6_address->sin6_addr, &in6addr_any,
               sizeof(struct in6_addr));                     /* auto-fill with local IP */
        inet6_address->sin6_scope_id = 0;
      } /* if */
#else
      if (unlikely(!ALLOC_BSTRI_SIZE_OK(result, sizeof(struct sockaddr_in)))) {
        raise_error(MEMORY_ERROR);
      } else {
        result->size = sizeof(struct sockaddr_in);
        inet_address = (struct sockaddr_in *) result->mem;
        inet_address->sin_family = AF_INET;
        inet_address->sin_port = htons((uint16Type) port); /* short, network byte order */
        inet_address->sin_addr.s_addr = INADDR_ANY;        /* auto-fill with local IP */
        memset(inet_address->sin_zero, '\0', sizeof(inet_address->sin_zero));
      } /* if */
#endif
#endif
    } /* if */
    logFunction(printf("socInetServAddr(" FMT_D ") --> \"%s\"\n",
                       port, socAddressCStri(result)););
    return result;
  } /* socInetServAddr */



#if HAS_POLL
boolType socInputReady (socketType sock, intType seconds, intType micro_seconds)

  {
    struct pollfd pollFd[1];
    int timeout;
    int poll_result;
    unsigned char next_char;
    memSizeType bytes_received;
    boolType inputReady;

  /* socInputReady */
    logFunction(printf("socInputReady(%d, " FMT_D ", " FMT_D ")\n",
                       sock, seconds, micro_seconds););
    if (unlikely(sock == (socketType) -1)) {
      logError(printf("socInputReady(%d, " FMT_D ", " FMT_D "): "
                      "Invalid socket.\n",
                      sock, seconds, micro_seconds););
      raise_error(FILE_ERROR);
      inputReady = FALSE;
    } else if (unlikely(seconds < 0 || seconds >= INT_MAX / 1000 ||
                 micro_seconds < 0 || micro_seconds >= 1000000)) {
      logError(printf("socInputReady(%d, " FMT_D ", " FMT_D"): "
                      "seconds or micro_seconds not in allowed range.\n",
                      sock, seconds, micro_seconds););
      raise_error(RANGE_ERROR);
      inputReady = FALSE;
    } else {
      pollFd[0].fd = (int) sock;
      pollFd[0].events = POLLIN;
      timeout = (int) seconds * 1000 + (int) (micro_seconds / 1000);
      poll_result = os_poll(pollFd, 1, timeout);
      if (unlikely(poll_result < 0)) {
        logError(printf("socInputReady(%d, " FMT_D ", " FMT_D "): "
                        "os_poll([%d, POLLIN], 1, %d) failed:\n"
                        "%s=%d\nerror: %s\n",
                        sock, seconds, micro_seconds,
                        sock, timeout,
                        ERROR_INFORMATION););
        raise_error(FILE_ERROR);
        inputReady = FALSE;
      } else {
        inputReady = poll_result == 1 && (pollFd[0].revents & POLLIN);
        if (inputReady) {
          /* Verify that it is really possible to read at least one character */
          bytes_received = (memSizeType) recv((os_socketType) sock,
                                              cast_send_recv_data(&next_char), 1, MSG_PEEK);
          if (bytes_received != 1) {
            logMessage(printf("socInputReady: bytes_received=" FMT_U_MEM "\n",
                              bytes_received););
            inputReady = FALSE;
          } /* if */
        } /* if */
      } /* if */
    } /* if */
    logFunction(printf("socInputReady(%d, " FMT_D ", " FMT_D ") --> %d\n",
                       sock, seconds, micro_seconds, inputReady););
    return inputReady;
  } /* socInputReady */

#else



boolType socInputReady (socketType sock, intType seconds, intType micro_seconds)

  {
    int nfds;
    fd_set readfds;
    struct timeval timeout;
    int select_result;
    unsigned char next_char;
    memSizeType bytes_received;
    boolType inputReady;

  /* socInputReady */
    logFunction(printf("socInputReady(%d, " FMT_D ", " FMT_D ")\n",
                       sock, seconds, micro_seconds););
    if (unlikely(sock == (socketType) -1)) {
      logError(printf("socInputReady(%d, " FMT_D ", " FMT_D "): "
                      "Invalid socket.\n",
                      sock, seconds, micro_seconds););
      raise_error(FILE_ERROR);
      inputReady = FALSE;
    } else if (unlikely(seconds < 0 || seconds >= LONG_MAX ||
                 micro_seconds < 0 || micro_seconds >= 1000000)) {
      logError(printf("socInputReady(%d, " FMT_D ", " FMT_D"): "
                      "seconds or micro_seconds not in allowed range.\n",
                      sock, seconds, micro_seconds););
      raise_error(RANGE_ERROR);
      inputReady = FALSE;
    } else {
      FD_ZERO(&readfds);
      FD_SET((os_socketType) sock, &readfds);
      nfds = (int) sock + 1;
      timeout.tv_sec = (long int) seconds;
      timeout.tv_usec = (long int) micro_seconds;
      /* printf("select(%d, %d)\n", nfds, sock); */
      select_result = select(nfds, &readfds, NULL, NULL, &timeout);
      /* printf("select_result: %d\n", select_result); */
      if (unlikely(select_result < 0)) {
        logError(printf("socInputReady(%d, " FMT_D ", " FMT_D "): "
                        "select(%d, [%d], NULL, NULL, [" FMT_D ", " FMT_D "]) failed:\n"
                        "%s=%d\nerror: %s\n",
                        sock, seconds, micro_seconds,
                        nfds, sock, seconds, micro_seconds,
                        ERROR_INFORMATION););
        raise_error(FILE_ERROR);
        inputReady = FALSE;
      } else {
        inputReady = FD_ISSET((os_socketType) sock, &readfds);
        if (inputReady) {
          /* Verify that it is really possible to read at least one character */
          bytes_received = (memSizeType) recv((os_socketType) sock,
                                              cast_send_recv_data(&next_char), 1, MSG_PEEK);
          if (bytes_received != 1) {
            logMessage(printf("socInputReady: bytes_received=" FMT_U_MEM "\n",
                              bytes_received););
            inputReady = FALSE;
          } /* if */
        } /* if */
      } /* if */
    } /* if */
    logFunction(printf("socInputReady(%d, " FMT_D ", " FMT_D ") --> %d\n",
                       sock, seconds, micro_seconds, inputReady););
    return inputReady;
  } /* socInputReady */

#endif



#define BUFFER_START_SIZE 256
#define BUFFER_DELTA_SIZE 256

/**
 *  Read a line from 'inSocket'.
 *  The function accepts lines ending with "\n", "\r\n" or EOF.
 *  The line ending characters are not copied into the string.
 *  That means that the "\r" of a "\r\n" sequence is silently removed.
 *  When the function is left terminationChar contains '\n' or EOF.
 *  @return the line read.
 *  @exception MEMORY_ERROR Not enough memory to represent the result.
 */
striType socLineRead (socketType inSocket, charType *const terminationChar)

  {
    memSizeType bytes_received;
    memSizeType bytes_requested;
    memSizeType result_size;
    memSizeType old_result_size;
    memSizeType result_pos;
    ucharType *nlPos;
    ucharType buffer[BUFFER_SIZE];
    striType resized_result;
    striType result;

  /* socLineRead */
    logFunction(printf("socLineRead(%d, '\\" FMT_U32 ";')\n",
                       inSocket, *terminationChar););
    if (unlikely(inSocket == (socketType) -1)) {
      logError(printf("socLineRead(%d, '\\" FMT_U32 ";'): Invalid socket.\n",
                      inSocket, *terminationChar););
      raise_error(FILE_ERROR);
      result = NULL;
    } else {
      bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                          cast_send_recv_data(buffer),
                                          cast_buffer_len(BUFFER_START_SIZE),
                                          MSG_PEEK);
      logMessage(printf("socLineRead: bytes_received=" FMT_U_MEM "\n",
                        bytes_received););
      if (unlikely(bytes_received == (memSizeType) -1)) {
        logError(printf("socLineRead: "
                        "recv(%d, *, " FMT_U_MEM ", MSG_PEEK) failed:\n"
                        "%s=%d\nerror: %s\n",
                        inSocket, (memSizeType) BUFFER_START_SIZE,
                        ERROR_INFORMATION););
        bytes_received = 0;
      } /* if */
      if (bytes_received == 0) {
        if (unlikely(!ALLOC_STRI_SIZE_OK(result, 0))) {
          raise_error(MEMORY_ERROR);
          result = NULL;
        } else {
          result->size = 0;
          *terminationChar = (charType) EOF;
        } /* if */
      } else {
        nlPos = (ucharType *) memchr(buffer, '\n', bytes_received);
        if (nlPos != NULL) {
          bytes_requested = (memSizeType) (nlPos - buffer) + 1;
          /* This should overwrite the buffer with identical data up to '\n'. */
          bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                              cast_send_recv_data(buffer),
                                              cast_buffer_len(bytes_requested), 0);
          /* bytes_received should always be identical to bytes_requested. */
          result_size = bytes_requested - 1;
          if (nlPos != buffer && nlPos[-1] == '\r') {
            result_size--;
          } /* if */
          if (unlikely(!ALLOC_STRI_CHECK_SIZE(result, result_size))) {
            raise_error(MEMORY_ERROR);
            result = NULL;
          } else {
            memcpy_to_strelem(result->mem, buffer, result_size);
            result->size = result_size;
            *terminationChar = '\n';
          } /* if */
        } else {
          result_size = bytes_received;
          old_result_size = 0;
          result_pos = 0;
          result = NULL;
          do {
            bytes_requested = bytes_received;
            /* This should overwrite the buffer with identical data. */
            bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                                cast_send_recv_data(buffer),
                                                cast_buffer_len(bytes_requested), 0);
            /* bytes_received should always be identical to bytes_requested. */
            result_size += BUFFER_DELTA_SIZE;
            /* printf("A result=%08lx, old_result_size=%d, result_size=%d\n",
                (unsigned long) result, old_result_size, result_size); */
            REALLOC_STRI_CHECK_SIZE2(resized_result, result, old_result_size, result_size);
            /* printf("B result=%08lx, resized_result=%08lx\n",
                (unsigned long) result, (unsigned long) resized_result); */
            if (unlikely(resized_result == NULL)) {
              if (result != NULL) {
                FREE_STRI2(result, old_result_size);
              } /* if */
              raise_error(MEMORY_ERROR);
              result = NULL;
            } else {
              result = resized_result;
              old_result_size = result_size;
              /* printf("a result[%d], size=%d\n", result_pos, bytes_requested); */
              memcpy_to_strelem(&result->mem[result_pos], buffer, bytes_requested);
              result_pos += bytes_requested;
              bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                                  cast_send_recv_data(buffer),
                                                  cast_buffer_len(BUFFER_DELTA_SIZE),
                                                  MSG_PEEK);
              if (unlikely(bytes_received == (memSizeType) -1)) {
                logError(printf("socLineRead: "
                                "recv(%d, *, " FMT_U_MEM ", MSG_PEEK) failed:\n"
                                "%s=%d\nerror: %s\n",
                                inSocket, (memSizeType) BUFFER_DELTA_SIZE,
                                ERROR_INFORMATION););
                bytes_received = 0;
              } /* if */
              if (bytes_received == 0) {
                REALLOC_STRI_CHECK_SIZE2(resized_result, result, result_size, result_pos);
                if (unlikely(resized_result == NULL)) {
                  FREE_STRI2(result, result_size);
                  raise_error(MEMORY_ERROR);
                  result = NULL;
                } else {
                  result = resized_result;
                  result->size = result_pos;
                  *terminationChar = (charType) EOF;
                } /* if */
              } else {
                nlPos = (ucharType *) memchr(buffer, '\n', bytes_received);
              } /* if */
            } /* if */
          } while (result != NULL && bytes_received != 0 && nlPos == NULL);
          if (result != NULL && nlPos != NULL) {
            bytes_requested = (memSizeType) (nlPos - buffer) + 1;
            /* This should overwrite the buffer with identical data up to '\n'. */
            bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                                cast_send_recv_data(buffer),
                                                cast_buffer_len(bytes_requested), 0);
            /* bytes_received should always be identical to bytes_requested. */
            bytes_requested--;
            if (nlPos == buffer) {
              if (result->mem[result_pos - 1] == '\r') {
                result_pos--;
              } /* if */
            } else if (nlPos[-1] == '\r') {
              bytes_requested--;
            } /* if */
            old_result_size = result_size;
            result_size = result_pos + bytes_requested;
            /* printf("C result=%08lx, old_result_size=%d, result_size=%d\n",
                (unsigned long) result, old_result_size, result_size); */
            REALLOC_STRI_CHECK_SIZE2(resized_result, result, old_result_size, result_size);
            /* printf("D result=%08lx, resized_result=%08lx\n",
                (unsigned long) result, (unsigned long) resized_result); */
            if (unlikely(resized_result == NULL)) {
              FREE_STRI2(result, result_size);
              raise_error(MEMORY_ERROR);
              result = NULL;
            } else {
              result = resized_result;
              /* printf("e result[%d], size=%d\n", result_pos, bytes_requested); */
              memcpy_to_strelem(&result->mem[result_pos], buffer, bytes_requested);
              result->size = result_size;
              *terminationChar = '\n';
            } /* if */
          } /* if */
        } /* if */
      } /* if */
    } /* if */
    logFunction(printf("socLineRead(%d, '\\" FMT_U32 ";') --> \"%s\"\n",
                       inSocket, *terminationChar, striAsUnquotedCStri(result)););
    return result;
  } /* socLineRead */



/**
 *  Listen for socket connections and limit the incoming queue.
 *  The 'backlog' argument defines the maximum length to which
 *  the queue of pending connections for 'listenerSocket' may grow.
 *  @exception FILE_ERROR A system function returns an error.
 */
void socListen (socketType listenerSocket, intType backlog)

  { /* socListen */
    logFunction(printf("socListen(%d, " FMT_D ")\n", listenerSocket, backlog););
    if (!inIntRange(backlog)) {
      logError(printf("socListen(%d, " FMT_D "): "
                      "backlog not in allowed range.\n",
                      listenerSocket, backlog););
      raise_error(RANGE_ERROR);
    } else if (unlikely(listen((os_socketType) listenerSocket,
                               (int) backlog) != 0)) {
      logError(printf("socListen: listen(%d, " FMT_D ") failed:\n"
                      "%s=%d\nerror: %s\n",
                      listenerSocket, backlog, ERROR_INFORMATION););
      raise_error(FILE_ERROR);
    } /* if */
  } /* socListen */



intType socRecv (socketType sock, striType *stri, intType length, intType flags)

  {
    striType resized_stri;
    memSizeType bytes_requested;
    memSizeType old_stri_size;
    memSizeType new_stri_size;

  /* socRecv */
    logFunction(printf("socRecv(%d, *, " FMT_D ", 0x" FMT_X ")\n",
                       sock, length, flags););
    if (unlikely(sock == (socketType) -1)) {
      logError(printf("socRecv(%d, *, " FMT_D ", 0x" FMT_X "): "
                      "Invalid socket.\n",
                      sock, length, flags););
      raise_error(FILE_ERROR);
      return 0;
    } else if (unlikely(length < 0 || !inIntRange(flags))) {
      logError(printf("socRecv(%d, *, " FMT_D ", 0x" FMT_X "): "
                      "length or flags not in allowed range.\n",
                      sock, length, flags););
      raise_error(RANGE_ERROR);
      return 0;
    } else {
      if ((uintType) length > MAX_MEMSIZETYPE) {
        bytes_requested = MAX_MEMSIZETYPE;
      } else {
        bytes_requested = (memSizeType) length;
      } /* if */
      old_stri_size = (*stri)->size;
      if (old_stri_size < bytes_requested) {
        REALLOC_STRI_CHECK_SIZE2(resized_stri, *stri, old_stri_size, bytes_requested);
        if (unlikely(resized_stri == NULL)) {
          raise_error(MEMORY_ERROR);
          return 0;
        } /* if */
        *stri = resized_stri;
        old_stri_size = bytes_requested;
      } /* if */
      new_stri_size = (memSizeType) recv((os_socketType) sock,
                                         cast_send_recv_data((*stri)->mem),
                                         cast_buffer_len(bytes_requested), (int) flags);
      if (likely(new_stri_size != (memSizeType) -1)) {
        memcpy_to_strelem((*stri)->mem, (ustriType) (*stri)->mem, new_stri_size);
        (*stri)->size = new_stri_size;
        if (new_stri_size < old_stri_size) {
          REALLOC_STRI_SIZE_OK2(resized_stri, *stri, old_stri_size, new_stri_size);
          if (unlikely(resized_stri == NULL)) {
            raise_error(MEMORY_ERROR);
            return 0;
          } /* if */
          *stri = resized_stri;
        } /* if */
      } /* if */
    } /* if */
    return (intType) new_stri_size;
  } /* socRecv */



intType socRecvfrom (socketType sock, striType *stri, intType length, intType flags,
    bstriType *address)

  {
    striType resized_stri;
    memSizeType old_address_size;
    bstriType resized_address;
    memSizeType bytes_requested;
    sockLenType addrlen;
    memSizeType stri_size;

  /* socRecvfrom */
    logFunction(printf("socRecvfrom(%d, *, " FMT_D ", 0x" FMT_X ")\n",
                       sock, length, flags););
    if (unlikely(sock == (socketType) -1)) {
      logError(printf("socRecvfrom(%d, *, " FMT_D ", 0x" FMT_X "): "
                      "Invalid socket.\n",
                      sock, length, flags););
      raise_error(FILE_ERROR);
      return 0;
    } else if (unlikely(length < 0 || !inIntRange(flags))) {
      logError(printf("socRecvfrom(%d, *, " FMT_D ", 0x" FMT_X "): "
                      "length or flags not in allowed range.\n",
                      sock, length, flags););
      raise_error(RANGE_ERROR);
      return 0;
    } else {
      if ((uintType) length > MAX_MEMSIZETYPE) {
        bytes_requested = MAX_MEMSIZETYPE;
      } else {
        bytes_requested = (memSizeType) length;
      } /* if */
      REALLOC_STRI_CHECK_SIZE(resized_stri, *stri, bytes_requested);
      if (unlikely(resized_stri == NULL)) {
        raise_error(MEMORY_ERROR);
        return 0;
      } /* if */
      *stri = resized_stri;
      old_address_size = (*address)->size;
      REALLOC_BSTRI_SIZE_OK(resized_address, *address, old_address_size, MAX_ADDRESS_SIZE);
      if (unlikely(resized_address == NULL)) {
        stri_size = (*stri)->size;
        REALLOC_STRI_SIZE_OK2(resized_stri, *stri, bytes_requested, stri_size);
        if (resized_stri == NULL) {
          (*stri)->size = bytes_requested;
        } else {
          *stri = resized_stri;
        } /* if */
        raise_error(MEMORY_ERROR);
        return 0;
      } else {
        *address = resized_address;
        COUNT3_BSTRI(old_address_size, MAX_ADDRESS_SIZE);
        addrlen = MAX_ADDRESS_SIZE;
        stri_size = (memSizeType) recvfrom((os_socketType) sock,
                                           cast_send_recv_data((*stri)->mem),
                                           cast_buffer_len(bytes_requested), (int) flags,
                                           (struct sockaddr *) (*address)->mem, &addrlen);
        if (unlikely(stri_size == (memSizeType) -1 ||
                     ADDRLEN_NEGATIVE(addrlen) ||
                     addrlen > MAX_ADDRESS_SIZE)) {
          REALLOC_BSTRI_SIZE_OK(resized_address, *address, MAX_ADDRESS_SIZE, old_address_size);
          if (resized_address == NULL) {
            (*address)->size = MAX_ADDRESS_SIZE;
          } else {
            *address = resized_address;
            COUNT3_BSTRI(MAX_ADDRESS_SIZE, old_address_size);
          } /* if */
          logError(printf("socRecvfrom: recvfrom(%d, ...) failed:\n"
                          "%s=%d\nerror: %s\n",
                          sock, ERROR_INFORMATION););
          raise_error(FILE_ERROR);
        } else {
          REALLOC_BSTRI_SIZE_OK(resized_address, *address, MAX_ADDRESS_SIZE,
                                (memSizeType) addrlen);
          if (unlikely(resized_address == NULL)) {
            (*address)->size = MAX_ADDRESS_SIZE;
            raise_error(MEMORY_ERROR);
            return 0;
          } else {
            *address = resized_address;
            COUNT3_BSTRI(MAX_ADDRESS_SIZE, (memSizeType) addrlen);
            (*address)->size = (memSizeType) addrlen;
          } /* if */
        } /* if */
      } /* if */
      if (likely(stri_size != (memSizeType) -1)) {
        memcpy_to_strelem((*stri)->mem, (ustriType) (*stri)->mem, stri_size);
        (*stri)->size = stri_size;
        if (stri_size < bytes_requested) {
          REALLOC_STRI_SIZE_OK2(resized_stri, *stri, bytes_requested, stri_size);
          if (unlikely(resized_stri == NULL)) {
            raise_error(MEMORY_ERROR);
            return 0;
          } /* if */
          *stri = resized_stri;
        } /* if */
      } /* if */
    } /* if */
    return (intType) stri_size;
  } /* socRecvfrom */



intType socSend (socketType sock, const const_striType stri, intType flags)

  {
    bstriType buf;
    memSizeType bytes_sent;
    errInfoType err_info = OKAY_NO_ERROR;
    intType result;

  /* socSend */
    logFunction(printf("socSend(%d, \"%s\", 0x" FMT_X ")\n",
                       sock, striAsUnquotedCStri(stri), flags););
    if (unlikely(sock == (socketType) -1)) {
      logError(printf("socSend(%d, \"%s\", 0x" FMT_X "): "
                      "Invalid socket.\n",
                      sock, striAsUnquotedCStri(stri), flags););
      raise_error(FILE_ERROR);
      result = 0;
    } else if (unlikely(!inIntRange(flags))) {
      logError(printf("socSend(%d, \"%s\", 0x" FMT_X "): "
                      "flags not in allowed range.\n",
                      sock, striAsUnquotedCStri(stri), flags););
      raise_error(RANGE_ERROR);
      result = 0;
    } else {
      buf = stri_to_bstri(stri, &err_info);
      if (unlikely(buf == NULL)) {
        logError(printf("socSend: stri_to_bstri(\"%s\", *) failed:\n"
                        "err_info=%d\n",
                        striAsUnquotedCStri(stri), err_info););
        raise_error(err_info);
        result = 0;
      } else {
        bytes_sent = (memSizeType) send((os_socketType) sock,
                                        cast_send_recv_data(buf->mem),
                                        cast_buffer_len(buf->size), (int) flags);
        FREE_BSTRI(buf, buf->size);
        if (unlikely(bytes_sent == (memSizeType) -1)) {
          result = -1;
        } else if (unlikely(bytes_sent > MAX_MEM_INDEX)) {
          result = MAX_MEM_INDEX;
        } else {
          result = (intType) bytes_sent;
        } /* if */
      } /* if */
    } /* if */
    return result;
  } /* socSend */



intType socSendto (socketType sock, const const_striType stri, intType flags,
    const_bstriType address)

  {
    bstriType buf;
    memSizeType bytes_sent;
    errInfoType err_info = OKAY_NO_ERROR;
    intType result;

  /* socSendto */
    logFunction(printf("socSendto(%d, \"%s\", 0x" FMT_X ", \"%s\")\n",
                       sock, striAsUnquotedCStri(stri), flags,
                       socAddressCStri(address)););
    if (unlikely(sock == (socketType) -1)) {
      logError(printf("socSendto(%d, \"%s\", 0x" FMT_X ", \"%s\"): "
                      "Invalid socket.\n",
                      sock, striAsUnquotedCStri(stri), flags,
                      socAddressCStri(address)););
      raise_error(FILE_ERROR);
      result = 0;
    } else if (unlikely(!inIntRange(flags))) {
      logError(printf("socSendto(%d, \"%s\", 0x" FMT_X ", \"%s\"): "
                      "flags not in allowed range.\n",
                      sock, striAsUnquotedCStri(stri), flags,
                      socAddressCStri(address)););
      raise_error(RANGE_ERROR);
      result = 0;
    } else {
      buf = stri_to_bstri(stri, &err_info);
      if (unlikely(buf == NULL)) {
        logError(printf("socSendto: stri_to_bstri(\"%s\", *) failed:\n"
                        "err_info=%d\n",
                        striAsUnquotedCStri(stri), err_info););
        raise_error(err_info);
        result = 0;
      } else {
        bytes_sent = (memSizeType) sendto((os_socketType) sock,
                                          cast_send_recv_data(buf->mem),
                                          cast_buffer_len(buf->size), (int) flags,
                                          (const struct sockaddr *) address->mem,
                                          (sockLenType) address->size);
        FREE_BSTRI(buf, buf->size);
        if (unlikely(bytes_sent == (memSizeType) -1)) {
          result = -1;
        } else if (unlikely(bytes_sent > MAX_MEM_INDEX)) {
          result = MAX_MEM_INDEX;
        } else {
          result = (intType) bytes_sent;
        } /* if */
      } /* if */
    } /* if */
    return result;
  } /* socSendto */



void socSetOptBool (socketType sock, intType optname, boolType optval)

  { /* socSetOptBool */
    logFunction(printf("socSetOptBool(%d, " FMT_D ", %s)\n",
                       sock, optname, optval ? "TRUE" : "FALSE"););
    switch (castIntTypeForSwitch(optname)) {
      case SOC_OPT_NONE:
        break;
      case SOC_OPT_REUSEADDR: {
          int so_reuseaddr = optval;
          if (unlikely(setsockopt((os_socketType) sock,
                                  SOL_SOCKET, SO_REUSEADDR,
                                  (const char *) &so_reuseaddr,
                                  sizeof(so_reuseaddr)) != 0)) {
            logError(printf("socSetOptBool(%d, " FMT_D ", %s): "
                            "setsockopt(%d, ...) failed:\n"
                            "%s=%d\nerror: %s\n",
                            sock, optname, optval ? "TRUE" : "FALSE",
                            sock, ERROR_INFORMATION););
            raise_error(FILE_ERROR);
          } /* if */
        }
        break;
      default:
        logError(printf("socSetOptBool(%d, " FMT_D ", %s): "
                        "Unsupported option.\n",
                        sock, optname, optval ? "TRUE" : "FALSE"););
        raise_error(RANGE_ERROR);
        break;
    } /* switch */
  } /* socSetOptBool */



socketType socSocket (intType domain, intType type, intType protocol)

  {
    os_socketType result;

  /* socSocket */
    logFunction(printf("socSocket(" FMT_D ", " FMT_D ", " FMT_D ")\n",
                       domain, type, protocol););
    if (unlikely(!inIntRange(domain) || !inIntRange(type) || !inIntRange(protocol))) {
      logError(printf("socSocket(" FMT_D ", " FMT_D ", " FMT_D "): "
                      "domain, type or protocol not in allowed range.\n",
                      domain, type, protocol););
      raise_error(RANGE_ERROR);
      result = 0;
    } else {
      /* printf("socSocket(%d, %d, %d)\n", domain, type, protocol); */
      check_initialization((socketType) -1);
      result = (os_socketType) socket((int) domain, (int) type, (int) protocol);
#if SOCKET_LIB == WINSOCK_SOCKETS && !TWOS_COMPLEMENT_INTTYPE
      /* In this case INVALID_SOCKET != (socketType) -1 holds and    */
      /* (socketType) -1 must be returned instead of INVALID_SOCKET. */
      /* Probably a computer, which needs this, does not exist.      */
      if (unlikely(result == INVALID_SOCKET)) {
        result = (os_socketType) -1;
      } /* if */
#endif
    } /* if */
    logFunction(printf("socSocket(" FMT_D ", " FMT_D ", " FMT_D ") --> %d\n",
                       domain, type, protocol, result););
    return (socketType) result;
  } /* socSocket */



/**
 *  Read a word from 'inSocket'.
 *  Before reading the word it skips spaces and tabs. The function
 *  accepts words ending with " ", "\t", "\n", "\r\n" or EOF.
 *  The word ending characters are not copied into the string.
 *  That means that the "\r" of a "\r\n" sequence is silently removed.
 *  When the function is left terminationChar contains ' ', '\t', '\n' or
 *  EOF.
 *  @return the word read.
 *  @exception MEMORY_ERROR Not enough memory to represent the result.
 */
striType socWordRead (socketType inSocket, charType *const terminationChar)

  {
    unsigned char ch;
    register memSizeType position;
    register memSizeType bytes_received;
    strElemType *memory;
    memSizeType memlength = READ_STRI_INIT_SIZE;
    memSizeType newmemlength;
    striType resized_result;
    striType result;

  /* socWordRead */
    logFunction(printf("socWordRead(%d, '\\" FMT_U32 ";')\n",
                       inSocket, *terminationChar););
    if (unlikely(inSocket == (socketType) -1)) {
      logError(printf("socWordRead(%d, '\\" FMT_U32 ";'): Invalid socket.\n",
                      inSocket, *terminationChar););
      raise_error(FILE_ERROR);
      result = NULL;
    } else if (unlikely(!ALLOC_STRI_SIZE_OK(result, memlength))) {
      raise_error(MEMORY_ERROR);
    } else {
      memory = result->mem;
      position = 0;
      do {
        bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                            cast_send_recv_data(&ch), 1, 0);
      } while (bytes_received == 1 && (ch == ' ' || ch == '\t'));
      while (bytes_received == 1 &&
          ch != ' ' && ch != '\t' && ch != '\n') {
        if (position >= memlength) {
          newmemlength = memlength + READ_STRI_SIZE_DELTA;
          REALLOC_STRI_CHECK_SIZE2(resized_result, result, memlength, newmemlength);
          if (unlikely(resized_result == NULL)) {
            FREE_STRI2(result, memlength);
            raise_error(MEMORY_ERROR);
            return NULL;
          } /* if */
          result = resized_result;
          memory = result->mem;
          memlength = newmemlength;
        } /* if */
        memory[position++] = (strElemType) ch;
        bytes_received = (memSizeType) recv((os_socketType) inSocket,
                                            cast_send_recv_data(&ch), 1, 0);
      } /* while */
      if (bytes_received == 1 && ch == '\n' &&
          position != 0 && memory[position - 1] == '\r') {
        position--;
      } /* if */
      REALLOC_STRI_SIZE_OK2(resized_result, result, memlength, position);
      if (unlikely(resized_result == NULL)) {
        FREE_STRI2(result, memlength);
        raise_error(MEMORY_ERROR);
        result = NULL;
      } else {
        result = resized_result;
        result->size = position;
        if (bytes_received != 1) {
          *terminationChar = (charType) EOF;
        } else {
          *terminationChar = (charType) ch;
        } /* if */
      } /* if */
    } /* if */
    logFunction(printf("socWordRead(%d, '\\" FMT_U32 ";') --> \"%s\"\n",
                       inSocket, *terminationChar, striAsUnquotedCStri(result)););
    return result;
  } /* socWordRead */



/**
 *  Write a string to 'outSocket'.
 *  @exception FILE_ERROR The system function is not able to write
 *             all characters of the string.
 *  @exception RANGE_ERROR The string contains a character that does
 *             not fit into a byte.
 */
void socWrite (socketType outSocket, const const_striType stri)

  {
    ucharType buffer[BUFFER_SIZE];
    ustriType buf;
    memSizeType bytes_to_send;
    memSizeType bytes_sent;
    memSizeType totally_sent = 0;
    errInfoType err_info = OKAY_NO_ERROR;
    bstriType bstri = NULL;

  /* socWrite */
    logFunction(printf("socWrite(%d, \"%s\")\n",
                       outSocket, striAsUnquotedCStri(stri)););
    if (unlikely(outSocket == (socketType) -1)) {
        logError(printf("socWrite(%d, \"%s\"): Invalid socket.\n",
                        outSocket, striAsUnquotedCStri(stri)););
      err_info = FILE_ERROR;
      buf = NULL;
      bytes_to_send = 0;
    } else if (stri->size <= BUFFER_SIZE) {
      if (unlikely(memcpy_from_strelem(buffer, stri->mem, stri->size))) {
        logError(printf("socWrite(%d, \"%s\"): "
                        "At least one character does not fit into a byte.\n",
                        outSocket, striAsUnquotedCStri(stri)););
        err_info = RANGE_ERROR;
        buf = NULL;
        bytes_to_send = 0;
      } else {
        buf = buffer;
        bytes_to_send = stri->size;
      } /* if */
    } else {
      bstri = stri_to_bstri(stri, &err_info);
      if (unlikely(bstri == NULL)) {
        logError(printf("socWrite(%d, \"%s\"): "
                        "Failed to create a temporary bstring.\n"
                        "err_info=%d\n",
                        outSocket, striAsUnquotedCStri(stri), err_info););
        buf = NULL;
        bytes_to_send = 0;
      } else {
        buf = bstri->mem;
        bytes_to_send = bstri->size;
      } /* if */
    } /* if */
    while (bytes_to_send != 0 && err_info == OKAY_NO_ERROR) {
      bytes_sent = (memSizeType) send((os_socketType) outSocket,
                                      cast_send_recv_data(&buf[totally_sent]),
                                      cast_buffer_len(bytes_to_send), 0);
      if (bytes_sent == (memSizeType) -1) {
        logError(printf("socWrite(%d, \"%s\" (size=" FMT_U_MEM ")): "
                        "send(%d, data, " FMT_U_MEM ") failed:\n"
                        "%s=%d\nerror: %s\n"
                        "bytes sent successfully: " FMT_U_MEM "\n",
                        outSocket, striAsUnquotedCStri(stri),
                        stri->size, outSocket, bytes_to_send,
                        ERROR_INFORMATION, totally_sent););
        err_info = FILE_ERROR;
      } else {
        totally_sent += bytes_sent;
        bytes_to_send -= bytes_sent;
      } /* if */
    } /* while */
    if (bstri != NULL) {
      FREE_BSTRI(bstri, bstri->size);
    } /* if */
    if (unlikely(err_info != OKAY_NO_ERROR)) {
      raise_error(err_info);
    } /* if */
  } /* socWrite */

#endif
